/**
 * Constructor for Logging class.
 * 
 * @param minimumSeverity the minimum severity that the logger should log. See static fields LEVEL_* of this class
 * @param writeLogFunction a function with two string arguments used for the severity and the actual log message
 * 
 * @returns {Logger}
 */
function Logger (minimumSeverity, writeLogFunction) {
	this.internalSeverity = minimumSeverity.level;
	this.writeLogFunction = writeLogFunction;
}

// Log levels
Logger.LEVEL_DEBUG = {"level" : 0, "text" : "DEBUG"};
Logger.LEVEL_INFO = {"level" : 1, "text" : "INFO"};
Logger.LEVEL_WARN = {"level" : 2, "text" : "WARN"};
Logger.LEVEL_ERROR = {"level" : 3, "text" : "ERROR"};

/**
 * Logs a message
 * 
 * @param messageSeverity the severity of the message
 * @param message the message
 */
Logger.prototype.log = function (messageSeverity, message) {
	if (!("level" in messageSeverity) || !("text" in messageSeverity)) {
		this.writeLogFunction(Logger.LEVEL_ERROR.text, "Severity contains no level or no text.");
		return;
	}
	if (messageSeverity.level >= this.internalSeverity) {
		this.writeLogFunction(messageSeverity.text, message);
	}
};

/**
 * Checks if the severity is equal or higher than the severity of the logger
 * 
 * @param minimumSeverity the severity to check against
 * 
 * @returns true, if the severity is equal or higher than the severity of the logger
 */
Logger.prototype.isSeverity = function (minimumSeverity) {
	if (!("level" in minimumSeverity)) {
		this.writeLogFunction(Logger.LEVEL_ERROR.text, "Severity contains no level.");
		return true;
	}
	return minimumSeverity.level <= this.internalSeverity;
};
	
Logger.prototype.debug = function(message) {
	this.log(Logger.LEVEL_DEBUG, message);
};

Logger.prototype.info = function(message) {
	this.log(Logger.LEVEL_INFO, message);
};

Logger.prototype.warn = function(message) {
	this.log(Logger.LEVEL_WARN, message);
};

Logger.prototype.error = function(message) {
	this.log(Logger.LEVEL_ERROR, message);
};

Logger.prototype.isDebug = function() {
	return this.isSeverity(Logger.LEVEL_DEBUG);
};

Logger.prototype.isInfo = function() {
	return this.isSeverity(Logger.LEVEL_INFO);
};

Logger.prototype.isWarn = function() {
	return this.isSeverity(Logger.LEVEL_WARN);
};

Logger.prototype.isError = function() {
	return this.isSeverity(Logger.LEVEL_ERROR);
};

Logger.prototype.setLevel = function(minimumSeverity) {
	this.internalSeverity = minimumSeverity.level;
};