/*
 * Main ReqIF sync script.
 * 
 * This script
 * - checks if the necessary custom properties are available in the project
 * - checks if the project was synced before, to do an import
 * - locks the project
 * - runs the import
 * - checks if an export is necessary either because the versions differ or because an import was run
 * - runs the export
 * - unlocks the project
 * 
 * It also writes a log file and an exit code if a log directory was specified
 */

/**
 * Connects to a HTTP Server and returns the status code and data
 * 
 * @param url URL to connect to
 * @param username username to use for authorization
 * @param password password to use for authorization
 * @param requestOptions options for the request
 * @param maxTries number of tries. Defaults to 20 if ommited. Should be at lest 1. <b>Attention:</b> functions in the requestOptions might get started once per request
 * 
 * @return Object with fields "code" and "content"
 * 
 * @throws exception if no connection could be established
 */
function sendRequestToServer(url, username, password, requestOptions, maxTries) {
	if (maxTries === undefined) {
		maxTries = 20;
	}
	var triesLeft = maxTries;
	var lastError = null;
	while (triesLeft > 0) {
		triesLeft--;
		try {
			var urlObject = new java.net.URL(url);
			var httpUrlConnection = urlObject.openConnection();
			if (typeof username === "string" || typeof password === "string" && username.length > 0 && password.length > 0) {
				var auth = new java.lang.String(username + ":" + password);
				var base64auth = javax.xml.bind.DatatypeConverter.printBase64Binary(auth.getBytes());
				httpUrlConnection.setRequestProperty("Authorization", "Basic " + base64auth);
			}
		
			// request options that need to be handled before the request is done
			if (requestOptions != null && typeof requestOptions === "object") {
				if (requestOptions.hasOwnProperty("requestMethod")) {
					httpUrlConnection.setRequestMethod(requestOptions["requestMethod"]);
				}
				if (requestOptions.hasOwnProperty("runBeforeRequest") && typeof requestOptions["runBeforeRequest"] === "function") {
					requestOptions.runBeforeRequest(httpUrlConnection);
				}
				if (requestOptions.hasOwnProperty("dataFunction") && typeof requestOptions["dataFunction"] === "function") {
					httpUrlConnection.setDoOutput(true);
					requestOptions.dataFunction(httpUrlConnection.getOutputStream()); 
				}
			}
		
			LOGGER.debug("Sending HTTP " + httpUrlConnection.getRequestMethod() + " request to " + url);
			
			var response = new Object();
			response.code = httpUrlConnection.getResponseCode();
			response.content = "";

			LOGGER.debug("Received HTTP response with status code " + httpUrlConnection.getResponseCode());
			
			// request options that need to be handled after the request is done
			if (requestOptions != null && typeof requestOptions === "object") {
				if (requestOptions.hasOwnProperty("runAfterRequest") && typeof requestOptions["runAfterRequest"] === "function") {
					requestOptions.runAfterRequest(httpUrlConnection);
				}
			}
			
			if (response.code == 200) {
				var reader = new java.io.BufferedReader(
					new java.io.InputStreamReader(
						httpUrlConnection.getInputStream(),
						"UTF-8"
					)
				);
				// reading contents of the response
				var content = new java.lang.StringBuilder(1024*512); // 512KB
				var chars = java.lang.reflect.Array.newInstance(java.lang.Character.TYPE, 8192);
				var nrCharsRead = -1;
				while ((nrCharsRead = reader.read(chars)) != -1) {
					content.append(chars, 0, nrCharsRead);
				}
				reader.close();
				response.content = String(content.toString());
			}
			return response;
		} catch (e) {
			lastError = e;
			LOGGER.warn("Exception '" + e.message + "' during HTTP request. Tries left: " + triesLeft + "/" + maxTries);
			java.lang.Thread.sleep(10000);
		}
	}
	throw lastError;
}

function URLBuilder(reqifBaseURL, coPilotBaseURL) {
	this.reqifurl = new java.net.URL(new java.net.URL(reqifBaseURL), "reqif/services/");
	this.ilahurl = new java.net.URL(new java.net.URL(coPilotBaseURL), "CockpitCoPilot/");
}

URLBuilder.prototype.getReqIFURL = function(path) {
	while(path.charAt(0) === '/' || path.charAt(0) === '\\') {
		path = path.substr(1);
	}
	return String(new java.net.URL(this.reqifurl, path).toString());
};

URLBuilder.prototype.getILAHURL = function(path) {
	while(path.charAt(0) === '/' || path.charAt(0) === '\\') {
		path = path.substr(1);
	}
	return String(new java.net.URL(this.ilahurl, path).toString());
};


// load libraries
load(new java.io.File(scriptFileParentDir, "lib/general/json2.js").getAbsolutePath());
load(new java.io.File(scriptFileParentDir, "lib/general/logger.js").getAbsolutePath());
load(new java.io.File(scriptFileParentDir, "lib/general/helpers.js").getAbsolutePath());
load(new java.io.File(scriptFileParentDir, "lib/general/cockpit.js").getAbsolutePath());

// load configuration
load(new java.io.File(scriptFileParentDir, "lib/reqif/ReqIF-conf-loader.js").getAbsolutePath());
var OPTION = (function() {
	if ("varargsJSONString" in cockpit) {
		try {
			var vararsgobj = JSON.parse(cockpit.varargsJSONString);
			return loadConf(vararsgobj);
		} catch (e) {
			throw new Error("Cannot parse varargsJSONString.");
		}
	} else {
		return loadConf(null);
	}
})();

// load identifiers for types and objects
load(new java.io.File(scriptFileParentDir, "lib/reqif/ReqIF-identifier.js").getAbsolutePath());

// load the import and export scripts
load(new java.io.File(scriptFileParentDir, "lib/reqif/ReqIF-import.js").getAbsolutePath());
load(new java.io.File(scriptFileParentDir, "lib/reqif/ReqIF-export.js").getAbsolutePath());

// Setting up the logger
var LOGGER;
(function() {
	var formatter = new java.text.SimpleDateFormat("yyyy-MM-dd HH:mm:ss,SSS");
	
	// line writer for either writing to a log file or to standard out via print 
	var lineWriter;
	if (cockpit.logDir != null) {
		var file = new java.io.File(cockpit.logDir + java.io.File.separator + "sync.log");
		var writer = new java.io.FileWriter(file);
		var printWriter = new java.io.PrintWriter(writer);

		lineWriter = function(line) {
			printWriter.println(line);
			printWriter.flush();
		};
	} else {
		lineWriter = function(line) {
			print(line);
		};
	}

	// function for formatting and writing log lines with the line writer defined before
	var writeFunction = function(severity, text) {
		if (severity.length < 5) {
			severity = Array(5 + 1 - severity.length).join(" ") + severity;
		}
		lineWriter(severity + " " + formatter.format(new java.util.Date()) + " [js]: " + text);
		
	};
	LOGGER = new Logger(Logger.LEVEL_DEBUG, writeFunction);

	var level = Logger.LEVEL_INFO;
	if (OPTION.LoggingLevel === "debug") {
		level = Logger.LEVEL_DEBUG;
	} else if (OPTION.LoggingLevel === "warn" || OPTION.LoggingLevel === "warning") {
		level = Logger.LEVEL_WARN;
	} else if (OPTION.LoggingLevel === "error") {
		level = Logger.LEVEL_ERROR;
	}
	LOGGER.info("Logging set up. Switching to minimum level '" + level.text + "'");
	LOGGER.setLevel(level);
})();

// writing configuration to log
if (LOGGER.isDebug()) {
	LOGGER.debug("Using following configuration options:");
	for (var key in OPTION) {
		if (OPTION.hasOwnProperty(key)) {
			if (key.toLowerCase().indexOf("password") != -1) {
				LOGGER.debug("conf: " + key + " = \"***\"");
			} else {
				LOGGER.debug("conf: " + key + " = \"" + OPTION[key] + "\"");
			}
		}
	}
}

if (OPTION.CoPilotBaseURL == "") {
	LOGGER.info("No seperate CoPilotBaseURL specified. Using value of ReqIFHost.");
	OPTION.CoPilotBaseURL = OPTION.ReqIFHost;
}

//object for storing global information that can be used in export and import
var global = new Object();

//URLs for ReqIF Sever and iLAH
global.urlbuilder = new URLBuilder(OPTION.ReqIFHost, OPTION.CoPilotBaseURL);

//Object with the project itself and information about the project
global.project = new Object();
global.project.project = cockpit.projectDataProvider.getProject();
global.project.reqIfIdentifier = IDENTIFIER.object.project + "-" + global.project.project.getUniqueIdentifier();
global.project.reqIfUrl = global.urlbuilder.getReqIFURL("projects/" + global.project.reqIfIdentifier);

//load language file

var langFile = "";
switch (String(global.project.project.getLanguage())) {
	case "de":
		langFile = "lib/reqif/ReqIF-lang_de.js";
		break;
	case "en":
		langFile = "lib/reqif/ReqIF-lang_en.js";
		break;
}
if (langFile !== "") {
	LOGGER.debug("Language file found for project language '" + global.project.project.getLanguage() + "'. Using '" + langFile + "'.");
} else {
	langFile = "lib/reqif/ReqIF-lang_en.js";
	LOGGER.warn("No language file found for project language '" + global.project.project.getLanguage() + "'. Using '" + langFile + "'.");	
}
load(new java.io.File(scriptFileParentDir, langFile).getAbsolutePath());

try {
	var schema = cockpit.schema;
	
	LOGGER.info("Checking custom properties and categories.");
	// checking conditions (required categories and custom properties exist)
	(function(){
	
		// check if the category exists
		if ( ! cockpit.dataModelProvider.hasCategory("gm.issuemodule2.issue", OPTION.IssueCategoryForComments) ) {
			throw new Error("Issue category '" + OPTION.IssueCategoryForComments + "' for comments not found.");
		}
		
		// checking if all necessary custom properties exist
		var optionsToCheck = [
		                      ["CustomPropertyReqIFIdentifier", "string"],
		                      ["CustomPropertyCreatedAt", "string"],
		                      ["CustomPropertyCreatedBy", "string"],
		                      ["CustomPropertyReferredElement", "string"]
	                         ];
		
		var customPropertiesForComments = cockpit.dataModelProvider.getCustomPropertiesForObjectTypeCategory(
				"gm.issuemodule2.issue", OPTION.IssueCategoryForComments);
		
		optionsToCheck.forEach(function(optionToCheck){
			
			assert(OPTION.hasOwnProperty(optionToCheck[0]), "Mandatory option '" + optionToCheck + "' not defined.");
	
			var option = OPTION[optionToCheck[0]];
			
			assert(option.startsWith("custom."), "Human readable id of custom property in option '" + optionToCheck[0] + "' has to start with 'custom.' but is '" + option + "'.");
			
			// checking for issues
			assert(schema.objecttypes["gm.issuemodule2.issue"].hasOwnProperty(option),
					"Custom property '" + option + "' specified in '" + optionToCheck[0] + "' does not exist for issues.");
			assert(customPropertiesForComments.indexOf(option) > -1,
					"Custom property '" + option + "' specified in '" + optionToCheck[0] + "' does not exist for issue category '" + OPTION.IssueCategoryForComments + "'.");
			var dataType = cockpit.dataModelProvider.getDatatypeIDOfProperty("gm.issuemodule2.issue", option);
			assert(dataType == optionToCheck[1],
					"Incorrect data type of custom property'" + option + "' for issues. Got '" + dataType + "' but expected '" + optionToCheck[1] + "'.");

			// checking for requirements
			assert(schema.objecttypes["gm.requirementsmodule3.requirement"].hasOwnProperty(option),
					"Custom property '" + option + "' specified in '" + optionToCheck[0] + "' does not exist for requirements.");
			dataType = cockpit.dataModelProvider.getDatatypeIDOfProperty("gm.requirementsmodule3.requirement", option);
			assert(dataType == optionToCheck[1],
					"Incorrect data type of custom property'" + option + "' for requirements. Got '" + dataType + "' but expected '" + optionToCheck[1] + "'.");
		});
		
		// checking if visibility flag exists and "makes sense"
		if (OPTION.hasOwnProperty("CustomPropertyExportVisibility") && typeof OPTION.ExportObjectsWithoutExportVisibilityCustomProperty === "boolean") {
			if (!OPTION.ExportObjectsWithoutExportVisibilityCustomProperty && !schema.objecttypes["gm.issuemodule2.issue"].hasOwnProperty(OPTION.CustomPropertyExportVisibility)) {
				LOGGER.warn("ExportObjectsWithoutExportVisibilityCustomProperty is 'false' and no custom property for visibility flag for issues found. Issues won't be visible after sync!");
			}
			if (!OPTION.ExportObjectsWithoutExportVisibilityCustomProperty && !schema.objecttypes["gm.issuemodule2.issueSet"].hasOwnProperty(OPTION.CustomPropertyExportVisibility)) {
				LOGGER.warn("ExportObjectsWithoutExportVisibilityCustomProperty is 'false' and no custom property for visibility flag for issue sets found. Issues won't be visible after sync!");
			}
			if (!OPTION.ExportObjectsWithoutExportVisibilityCustomProperty && customPropertiesForComments.indexOf(OPTION.CustomPropertyExportVisibility) == -1) {
				LOGGER.warn("ExportObjectsWithoutExportVisibilityCustomProperty is 'false' and no custom property for visibility flag for issue category '" + OPTION.IssueCategoryForComments + "' found. " +
						"Comments won't be visible after sync!");
			}
		}
	})();
	
	// Checking if connection to ReqIF server can be established by getting the projects list
	LOGGER.info("Checking if ReqIF Server is reachable");
	var allProjectResponse = sendRequestToServer(global.urlbuilder.getReqIFURL("projects.json"), OPTION.ReqIFUsername, OPTION.ReqIFPassword, null);
	assert(allProjectResponse.code === 200, "Cannot connect to ReqIF Server. Initial connection test failed with code '" + allProjectResponse.code + "'.");

	// getting the current version of the project on the cockpit server
	global.project.maxVersion = (function() {
		var maxVersion = 0;
		global.project.project.getProjectHistoryEntries().forEach(function(entry){
			var version = Number(entry.getVersion());
			if (version > maxVersion) {
				maxVersion = version;
			}
		});
		return maxVersion;
	})();
	
	var caughtError;
	var lockAccquired;

	var doImport = true;
	var versionOnServer = -1;
	
	// Checking if the project was exported before and what version
	(function(){
		LOGGER.debug("Checking if project was exported before.");
		var response = sendRequestToServer(global.project.reqIfUrl + ".json", OPTION.ReqIFUsername, OPTION.ReqIFPassword, null);
		if (response.code === 200 && response.content) {
			LOGGER.info("Project found on server.");
			LOGGER.debug("Checking version of exported project.");
			response = sendRequestToServer(
					global.project.reqIfUrl + "/specObjects/Setting-ProjectRevision-" + global.project.reqIfIdentifier + ".json",
					OPTION.ReqIFUsername,
					OPTION.ReqIFPassword,
					null
				);
			
			// if the project exists, the version information has to exist too, otherwise something is wrong and the sync should be stopped
			assert (response.code === 200 && response.content, "Project exists but no revision information available.");
			
			var currentRevisionOnServer = JSON.parse(response.content);
			
			assert (currentRevisionOnServer.hasOwnProperty("values") && Array.isArray(currentRevisionOnServer["values"]), "No 'value' array found in server response.");
			assert (currentRevisionOnServer["values"].length === 1, "Expected exactly one value. Got " + currentRevisionOnServer["values"].length + " value.");
			assert (currentRevisionOnServer["values"][0].hasOwnProperty("theValue"), "Value has no 'theValue' property");
			
			LOGGER.debug("Version on ReqIF " + Number(currentRevisionOnServer["values"][0]["theValue"]) + " <-> " + global.project.maxVersion + " Version in Cockpit");
			
			versionOnServer = Number(currentRevisionOnServer["values"][0]["theValue"]);
		} else if (response.code === 404) {
			LOGGER.info("Project not found. Not trying to import data from ReqIF Server.");
			doImport = false;
		} else {
			throw new Error("Project status on ReqIF Server unknown. Reponse from Server " + JSON.stringify(response));
		}
	})();
	
	var oldUsers = [];	
	var importDone = false;
	var projectLocked = false;
	if (doImport) {
		// lock Project
		LOGGER.debug("Locking project");
		var lockResponse = sendRequestToServer(global.project.reqIfUrl + "/lock.xml", OPTION.ReqIFUsername, OPTION.ReqIFPassword, {"requestMethod" : "PUT"});
		assert(lockResponse.code === 200, "Cannot lock project. Response code '" + lockResponse.code + "'");
		projectLocked = true;
		
		LOGGER.debug("Calling main function of import script.");
		importDone = startImportFromReqIF();
	} else {
		LOGGER.debug("Not calling main function of import script.");
	}
	
	// run the export if there already is a version difference or if there will be one due to a just done import
	if (global.project.maxVersion > versionOnServer || importDone) {
		
		LOGGER.debug("Calling main function of export script.");
		startExportToReqIF(importDone);
		
		/* ===== Create the annotator role and set "create" permissions ====== */
		(function() {

			LOGGER.debug("Checking for role '" + OPTION.ReqIFRole + "' in project.");
			
			// Adding the role if it doesn't exist
			var rolesResponse = sendRequestToServer(
					global.urlbuilder.getReqIFURL("/projects/" + global.project.reqIfIdentifier + "/roles.json"),
					OPTION.ReqIFUsername,
					OPTION.ReqIFPassword,
					null
			);
			
			assert(rolesResponse.code == 200, "Could not retrieve roles for project.");
			
			var roles = JSON.parse(rolesResponse.content);
			
			assert(roles.hasOwnProperty("projectRoles"), "Missing property 'projectRoles'.");
			
			if (roles["projectRoles"].indexOf(OPTION.ReqIFRole) === -1) {
				
				// The SpecObjects and permissions
				var specTypes = [
				                 "OT-" + IDENTIFIER.object.newComment,
				                 "OT-" + IDENTIFIER.object.newIssue,
				                 "OT-" + IDENTIFIER.object.newRequirement,
				                 "RT-" + IDENTIFIER.relation.CommentAnnotatesObject,
				                 "RT-" + IDENTIFIER.relation.NewIssue,
				                 "RT-" + IDENTIFIER.relation.NewRequirement
				                 ];
				var specTypePermissions = new Object();
				specTypes.forEach(function(specType) {
					var permission = new Object();
					if (specType === "OT-" + IDENTIFIER.object.newRequirement) { // new requirements are only readable if public
						permission["read"] = OPTION.NewRequirementsArePublic;
					} else {
						permission["read"] = true;
					}
					permission["update"] = permission["write"] = false;
					permission["create"] = true;
					permission["delete"] = false;
					permission["administer"] = false;
					specTypePermissions[specType] = permission;
				});
				
				LOGGER.debug("Adding role '" + OPTION.ReqIFRole + "' to project.");
				
				var createRoleResponse = sendRequestToServer(
						global.urlbuilder.getReqIFURL("/projects/" + global.project.reqIfIdentifier + "/roles/" + OPTION.ReqIFRole + ".json"),
						OPTION.ReqIFUsername,
						OPTION.ReqIFPassword,
						{"requestMethod" : "POST"}
				);
				if (createRoleResponse.code != 201) {
					throw new Error("Could not create role.");
				}
				
				LOGGER.debug("Setting permissions of role '" + OPTION.ReqIFRole + "' in project.");
				
				// Setting 'create' the permissions of the role for each spec type
				for (var specType in specTypePermissions) {
					var setPermissionForSpecTypeResponse = sendRequestToServer(
							global.urlbuilder.getReqIFURL("/projects/" + global.project.reqIfIdentifier + "/specTypes/" + specType + "/roles/" + OPTION.ReqIFRole + "/permissions.json"),
							OPTION.ReqIFUsername,
							OPTION.ReqIFPassword,
							{
								"requestMethod" : "POST",
								"dataFunction" : function(outputStream) {
									var outstreamwriter = new java.io.OutputStreamWriter(outputStream);
									outstreamwriter.write(JSON.stringify(specTypePermissions[specType]));
									outstreamwriter.flush();
									outstreamwriter.close();
								}
							}
					);
					
					if (setPermissionForSpecTypeResponse.code != 200) {
						throw new Error("Could not set permissions for SpecType '" + specType + "' and role '" + OPTION.ReqIFRole + "'.");
					}
				}
				
				// Setting 'read' the permissions of the role for the project
				var setPermissionForProjectResponse = sendRequestToServer(
						global.urlbuilder.getReqIFURL("/projects/" + global.project.reqIfIdentifier + "/roles/" + OPTION.ReqIFRole + "/permissions.json"),
						OPTION.ReqIFUsername,
						OPTION.ReqIFPassword,
						{
							"requestMethod" : "POST",
							"dataFunction" : function(outputStream) {
								var permission = new Object();
								permission["read"] = true;
								permission["update"] = false;
								permission["create"] = false;
								permission["delete"] = false;
								permission["administer"] = false;
								var outstreamwriter = new java.io.OutputStreamWriter(outputStream);
								outstreamwriter.write(JSON.stringify(permission));
								outstreamwriter.flush();
								outstreamwriter.close();
							}
						}
				);
				
				if (setPermissionForProjectResponse.code != 200) {
					throw new Error("Could not set permissions for SpecType '" + specType + "' and role '" + OPTION.ReqIFRole + "'.");
				}
			}
		})();
	} else {
		LOGGER.debug("Not calling main function of export script.");
	}

	// unlocking project
	if (projectLocked) {
		LOGGER.debug("Unlocking project");
		var unlockResponse = sendRequestToServer(global.project.reqIfUrl + "/unlock.xml", OPTION.ReqIFUsername, OPTION.ReqIFPassword, {"requestMethod" : "PUT"});
		if (unlockResponse.code !== 200) {
			LOGGER.warn("Cannot unlock project. Reponse code '" + unlockResponse.code + "'");
		}
	}

} catch(e) {
	caughtError = e;
	if ("fileName" in e && e.fileName !== null && "lineNumber" in e && e.lineNumber !== 0) {
		LOGGER.error(e + " (" + e.fileName + "#" + e.lineNumber +")");
	} else {
		LOGGER.error(e);
	}
}

// Writing file with exitcode
if (cockpit.logDir != null) {
	var exitCodeFile = new java.io.File(cockpit.logDir + java.io.File.separator + "exitcode.txt");
	var exitCodeWriter = new java.io.FileWriter(exitCodeFile);
	if (!caughtError) {
		exitCodeWriter.write("0");
	} else {
		exitCodeWriter.write("1");
	}
	exitCodeWriter.flush();
	exitCodeWriter.close();
}

// pass caught error to Java by rethrowing it 
if (caughtError) {
	throw caughtError;
}