/*
 * Import for Cockpit from ReqIF Server.
 * 
 * This script provides a function that
 * - retrieves all new comments, issues and requirements
 * - creates a repository snippet including the new objects, relations, parent folders and categories
 * - imports the objects into the local Cockpit client
 */

/**
 * Starts the import from the ReqIF Server to the local Cockpit client
 * 
 * @returns true if objects have been imported, false if not
 * 
 * @throws Error on unrecoverable error
 */
function startImportFromReqIF() {

/**
 * Parses a comment and creates all necessary explicit object and relations
 * 
 * @param comment the comment object to parse
 * @param parentreference the parent issue set that the created isses will be under
 * 
 * @return a repository snippet
 */
function parseComment(comment, parentreference) {
	if (LOGGER.isDebug()) {
		LOGGER.log(Logger.LEVEL_DEBUG, "Parsing comment. Object: " + JSON.stringify(comment));
	}
	var localSnippet = new Object();
	localSnippet.explicitobject = [];
	localSnippet.crosslinkrelation = [];
	
	var issueSchema = schema.objecttypes["gm.issuemodule2.issue"];

	var commentIdentifier = comment["identifier"];
	var commentText = null;
	
	if (global.IDsOfAlreadyImportedComments.indexOf(commentIdentifier) > -1) {
		LOGGER.info("Comment with ID '" + commentIdentifier + "' does already exist in the target folder. Skipping.");
		return localSnippet;
	}
	
	assert(Array.isArray(comment["values"]), "Values field does not exist or is no array.");
	
	comment["values"].forEach(function(value) {
		if (value["type"] == "ATTRIBUTE-VALUE-SIMPLE" && value["attributeDefinitionSimple"] == "AD-ObjectText-" + IDENTIFIER.object.newComment) {
			if (value["theValue"] !== null) {
				commentText = String(value["theValue"]);
			} else {
				commentText = "";
			}
		}
	});
	
	assert(commentText !== null, function() {return "Comment text not found. Object: " + JSON.stringify(comment);});
	
	var explicitobject = new Object();
	explicitobject.repositoryobjectsample = new Object();
	explicitobject.repositoryobjectsample.objecttypeid = issueSchema["objectTypeID"];
	explicitobject.repositoryobjectsample.category = OPTION.IssueCategoryForComments;
	explicitobject.repositoryobjectsample.properties = new Object();
	explicitobject.repositoryobjectsample.properties[issueSchema["uid"]] = String(cockpit.UUIDGenerator.getUniqueID());
	explicitobject.repositoryobjectsample.properties[issueSchema["id"]] = commentIdentifier;
	explicitobject.repositoryobjectsample.properties[issueSchema["description"]] = commentText;
	
	// adding additional req if specific custom properties
	addPropertyToSnippetObject(OPTION.CustomPropertyExportVisibility, "true", explicitobject.repositoryobjectsample.properties, issueSchema);
	if (comment.hasOwnProperty("createdAt")) {
		addPropertyToSnippetObject(OPTION.CustomPropertyCreatedAt, comment.createdAt, explicitobject.repositoryobjectsample.properties, issueSchema);
	}
	if (comment.hasOwnProperty("createdBy")) {
		addPropertyToSnippetObject(OPTION.CustomPropertyCreatedBy, comment.createdBy, explicitobject.repositoryobjectsample.properties, issueSchema);
	}
	
	var relatedObject = getSingleRelatedObject(commentIdentifier);
	
	if (relatedObject != null) {
		var parsedIdentifier = parseIdentifier(relatedObject["identifier"]);
		var relatedObjectHumanReadableText = getRelatedObjectAsHumanReadableText(relatedObject, parsedIdentifier);
		
		// Adding information about the related object to the explicit object
		addPropertyToSnippetObject(
			OPTION.CustomPropertyReferredElement,
			relatedObjectHumanReadableText,
			explicitobject.repositoryobjectsample.properties,
			issueSchema
		);
		
		var url = buildURLToObjektInILAH(relatedObject["identifier"]);
		addPropertyToSnippetObject(OPTION.CustomPropertyReferredElementURL, url, explicitobject.repositoryobjectsample.properties, issueSchema);

		// Adding the relation between the new issue and the cockpit object
		var relationInCockpitPossible = false;
		if (parsedIdentifier) {
			if (parsedIdentifier.schemaObjectType == "gm.requirementsmodule3.requirement" || parsedIdentifier.schemaObjectType == "gm.issuemodule2.issue") {
				crosslinkrelation = createCrossLinkRelationIssue_CrossModuleDataLinks(explicitobject.repositoryobjectsample.properties[issueSchema["uid"]], parsedIdentifier.objectUID);
				createContextObject(parsedIdentifier.schemaObjectType, parsedIdentifier.objectUID);
				localSnippet.crosslinkrelation.push(crosslinkrelation);
				relationInCockpitPossible = true;
			} else if (parsedIdentifier.schemaObjectType.startsWith("modelElement.")) {
				var crosslinkrelation = createCrossLinkRelationIssue_ModelElement(explicitobject.repositoryobjectsample.properties[issueSchema["uid"]], parsedIdentifier.objectUID);
				createContextObject(parsedIdentifier.schemaObjectType, parsedIdentifier.objectUID);
				localSnippet.crosslinkrelation.push(crosslinkrelation);
				relationInCockpitPossible = true;
			}
		} 
		
		explicitobject.repositoryobjectsample.properties[issueSchema["title"]] = LABEL.IssueTitleForReqIFCommentWithRelatedObject + " " + relatedObjectHumanReadableText;
		
		if (!relationInCockpitPossible) {
			addPropertyToSnippetObject(OPTION.CustomPropertyReqIFIdentifier, relatedObject["identifier"], explicitobject.repositoryobjectsample.properties, issueSchema);
			LOGGER.debug("Linked comment to \"" + parsedIdentifier.schemaObjectType + "\" via custom property");
		} else {
			LOGGER.debug("Linked comment to \"" + parsedIdentifier.schemaObjectType + "\" natively.");
		}
	} else {
		explicitobject.repositoryobjectsample.properties[issueSchema["title"]] = LABEL.IssueTitleForReqIFCommentWithoutRelatedObject;
	}
	
	explicitobject.parentobjectreference = parentreference;
	localSnippet.explicitobject.push(explicitobject);
	
	return localSnippet;
}

/**
 * Parses a new requirement and creates all necessary explicit objects and relations
 * 
 * @param newRequirement the new requirement object to parse
 * @param parentreference the parent requirement set that the created requirement will be under
 * 
 * @return a repository snippet
 */
function parseNewRequirement(newRequirement, parentreference) {
	if (LOGGER.isDebug()) {
		LOGGER.log(Logger.LEVEL_DEBUG, "Parsing new requirement. Object: " + JSON.stringify(newRequirement));
	}
	var localSnippet = new Object();
	localSnippet.explicitobject = [];
	localSnippet.crosslinkrelation = [];
	
	var reqSchema = schema.objecttypes["gm.requirementsmodule3.requirement"];

	var newRequirementIdentifier = newRequirement["identifier"];
	var newRequirementName = null;
	var newRequirementText = null;
	
	if (global.IDsOfAlreadyImportedNewRequirements.indexOf(newRequirementIdentifier) > -1) {
		LOGGER.info("New requirement with ID '" + newRequirementIdentifier + "' does already exist in target folder. Skipping.");
		return localSnippet;
	}
	
	assert(Array.isArray(newRequirement["values"]), "Values field does not exist or is no array.");
	
	newRequirement["values"].forEach(function(value) {
		if (value["type"] == "ATTRIBUTE-VALUE-SIMPLE" && value["attributeDefinitionSimple"] == "AD-ObjectHeading-" + IDENTIFIER.object.newRequirement) {
			newRequirementName = String(value["theValue"]).trim().replace(/\s/g, " ").trim();
			if (newRequirementName.length <= 0) {
				newRequirementName = "-";
			}
		} else if (value["type"] == "ATTRIBUTE-VALUE-SIMPLE" && value["attributeDefinitionSimple"] == "AD-ObjectText-" + IDENTIFIER.object.newRequirement) {
			if (value["theValue"] !== null) {
				newRequirementText = String(value["theValue"]);
			} else {
				newRequirementText = "";
			}
		} 
	});
	
	assert(newRequirementName !== null, function () {return "Requirement title not found. Object: " + JSON.stringify(newRequirement);});
	
	assert(newRequirementText !== null, function () {return "Requirement text not found. Object: " + JSON.stringify(newRequirement);});
	
	var explicitobject = new Object();
	explicitobject.repositoryobjectsample = new Object();
	explicitobject.repositoryobjectsample.objecttypeid = reqSchema["objectTypeID"];
	explicitobject.repositoryobjectsample.category = null;
	explicitobject.repositoryobjectsample.properties = new Object();
	explicitobject.repositoryobjectsample.properties[reqSchema["uid"]] = String(cockpit.UUIDGenerator.getUniqueID());
	explicitobject.repositoryobjectsample.properties[reqSchema["id"]] = newRequirementIdentifier;
	explicitobject.repositoryobjectsample.properties[reqSchema["name"]] = newRequirementName;
	explicitobject.repositoryobjectsample.properties[reqSchema["description"]] = newRequirementText; 
	
	// adding additional req if specific custom properties
	if (newRequirement.hasOwnProperty("createdAt")) {
		addPropertyToSnippetObject(OPTION.CustomPropertyCreatedAt, newRequirement.createdAt, explicitobject.repositoryobjectsample.properties, reqSchema);
	}
	if (newRequirement.hasOwnProperty("createdBy")) {
		addPropertyToSnippetObject(OPTION.CustomPropertyCreatedBy, newRequirement.createdBy, explicitobject.repositoryobjectsample.properties, reqSchema);
	}
	
	var relatedObject = getSingleRelatedObject(newRequirementIdentifier);
	
	if (relatedObject != null) {
		var parsedIdentifier = parseIdentifier(relatedObject["identifier"]);
		
		// Adding information about the related object to the explicit object
		addPropertyToSnippetObject(
			OPTION.CustomPropertyReferredElement,
			getRelatedObjectAsHumanReadableText(relatedObject, parsedIdentifier),
			explicitobject.repositoryobjectsample.properties,
			reqSchema
		);
		
		var url = buildURLToObjektInILAH(relatedObject["identifier"]);
		addPropertyToSnippetObject(OPTION.CustomPropertyReferredElementURL, url, explicitobject.repositoryobjectsample.properties, reqSchema);

		// Adding the relation between the new issue and the cockpit object
		var relationInCockpitPossible = false;
		if (parsedIdentifier) {
			if (parsedIdentifier.schemaObjectType.startsWith("modelElement.")) {
				var crosslinkrelation = createCrossLinkRelationRequirement_ModelElement(explicitobject.repositoryobjectsample.properties[reqSchema["uid"]], parsedIdentifier.objectUID);
				createContextObject(parsedIdentifier.schemaObjectType, parsedIdentifier.objectUID);
				localSnippet.crosslinkrelation.push(crosslinkrelation);
				relationInCockpitPossible = true;
			}
		}
		
		if (!relationInCockpitPossible) {
			addPropertyToSnippetObject(OPTION.CustomPropertyReqIFIdentifier, relatedObject["identifier"], explicitobject.repositoryobjectsample.properties, reqSchema);
			LOGGER.debug("Linked new requirement to \"" + parsedIdentifier.schemaObjectType + "\" via custom property");
		} else {
			LOGGER.debug("Linked new requirement to \"" + parsedIdentifier.schemaObjectType + "\" natively.");
		}
	}
	
	explicitobject.parentobjectreference = parentreference;
	localSnippet.explicitobject.push(explicitobject);
	
	return localSnippet;
}

/**
 * Parses a new issues and creates all necessary explicit object and relations
 * 
 * @param newIssue the new issue object to parse
 * @param parentreference the parent issue set that the created issue will be under
 * 
 * @return a repository snippet
 */
function parseNewIssue(newIssue, parentreference) {
	if (LOGGER.isDebug()) {
		LOGGER.log(Logger.LEVEL_DEBUG, "Parsing new issue. Object: " + JSON.stringify(newIssue));
	}
	var localSnippet = new Object();
	localSnippet.explicitobject = [];
	localSnippet.crosslinkrelation = [];
	
	var issueSchema = schema.objecttypes["gm.issuemodule2.issue"];

	var newIssueIdentifier = newIssue["identifier"];
	var newIssueTitle = null;
	var newIssueText = null;
	
	if (global.IDsOfAlreadyImportedNewIssues.indexOf(newIssueIdentifier) > -1) {
		LOGGER.info("New requirement with ID '" + newIssueIdentifier + "' does already exist in target folder. Skipping.");
		return localSnippet;
	}
	
	assert(Array.isArray(newIssue["values"]), function () {return "Values field does not exist or is no array. Object: " + JSON.stringify(newIssue);});
	
	newIssue["values"].forEach(function(value) {
		if (value["type"] == "ATTRIBUTE-VALUE-SIMPLE" && value["attributeDefinitionSimple"] == "AD-ObjectHeading-" + IDENTIFIER.object.newIssue) {
			newIssueTitle = String(value["theValue"]).trim().replace(/\s/g, " ").trim();
			if (newIssueTitle.length <= 0) {
				newIssueTitle = "-";
			}
		} else if (value["type"] == "ATTRIBUTE-VALUE-SIMPLE" && value["attributeDefinitionSimple"] == "AD-ObjectText-" + IDENTIFIER.object.newIssue) {
			if (value["theValue"] !== null) {
				newIssueText = String(value["theValue"]);
			} else {
				newIssueText = "";
			}
		} 
	});

	assert (newIssueTitle !== null, function () {return "Issue title not found. Object: " + JSON.stringify(newIssue);});
	
	assert (newIssueText !== null, function () {return"Issue text not found. Object: " + JSON.stringify(newIssue);});
	
	var explicitobject = new Object();
	explicitobject.repositoryobjectsample = new Object();
	explicitobject.repositoryobjectsample.objecttypeid = issueSchema["objectTypeID"];
	explicitobject.repositoryobjectsample.category = null;
	explicitobject.repositoryobjectsample.properties = new Object();
	explicitobject.repositoryobjectsample.properties[issueSchema["uid"]] = String(cockpit.UUIDGenerator.getUniqueID());
	explicitobject.repositoryobjectsample.properties[issueSchema["id"]] = newIssueIdentifier;
	explicitobject.repositoryobjectsample.properties[issueSchema["title"]] = newIssueTitle;
	explicitobject.repositoryobjectsample.properties[issueSchema["description"]] = newIssueText;
	
	// adding additional req if specific custom properties
	addPropertyToSnippetObject(OPTION.CustomPropertyExportVisibility, "true", explicitobject.repositoryobjectsample.properties, issueSchema);
	if (newIssue.hasOwnProperty("createdAt")) {
		addPropertyToSnippetObject(OPTION.CustomPropertyCreatedAt, newIssue.createdAt, explicitobject.repositoryobjectsample.properties, issueSchema);
	}
	if (newIssue.hasOwnProperty("createdBy")) {
		addPropertyToSnippetObject(OPTION.CustomPropertyCreatedBy, newIssue.createdBy, explicitobject.repositoryobjectsample.properties, issueSchema);
	}
	
	var relatedObject = getSingleRelatedObject(newIssueIdentifier);
	
	if (relatedObject != null) {
		var parsedIdentifier = parseIdentifier(relatedObject["identifier"]);
		
		// Adding information about the related object to the explicit object
		addPropertyToSnippetObject(
			OPTION.CustomPropertyReferredElement,
			getRelatedObjectAsHumanReadableText(relatedObject, parsedIdentifier),
			explicitobject.repositoryobjectsample.properties,
			issueSchema
		);
		
		var url = buildURLToObjektInILAH(relatedObject["identifier"]);
		addPropertyToSnippetObject(OPTION.CustomPropertyReferredElementURL, url, explicitobject.repositoryobjectsample.properties, issueSchema);
		
		// Adding the relation between the new issue and the cockpit object 
		var relationInCockpitPossible = false;
		if (parsedIdentifier) {
			if (parsedIdentifier.schemaObjectType == "gm.requirementsmodule3.requirement" || parsedIdentifier.schemaObjectType == "gm.issuemodule2.issue") {
				crosslinkrelation = createCrossLinkRelationIssue_CrossModuleDataLinks(explicitobject.repositoryobjectsample.properties[issueSchema["uid"]], parsedIdentifier.objectUID);
				createContextObject(parsedIdentifier.schemaObjectType, parsedIdentifier.objectUID);
				localSnippet.crosslinkrelation.push(crosslinkrelation);
				relationInCockpitPossible = true;
			} else if (parsedIdentifier.schemaObjectType.startsWith("modelElement.")) {
				var crosslinkrelation = createCrossLinkRelationIssue_ModelElement(explicitobject.repositoryobjectsample.properties[issueSchema["uid"]], parsedIdentifier.objectUID);
				createContextObject(parsedIdentifier.schemaObjectType, parsedIdentifier.objectUID);
				localSnippet.crosslinkrelation.push(crosslinkrelation);
				relationInCockpitPossible = true;
			}
		}
		
		if (!relationInCockpitPossible) {
			addPropertyToSnippetObject(OPTION.CustomPropertyReqIFIdentifier, relatedObject["identifier"], explicitobject.repositoryobjectsample.properties, issueSchema);
			LOGGER.debug("Linked new issue to \"" + parsedIdentifier.schemaObjectType + "\" via custom property");
		} else {
			LOGGER.debug("Linked new issue to \"" + parsedIdentifier.schemaObjectType + "\" natively.");
		}
	}

	explicitobject.parentobjectreference = parentreference;
	localSnippet.explicitobject.push(explicitobject);
	
	return localSnippet;
}

/**
 * Returns the related object to an identifier. If more than one related object exists, only the first one is returned
 * 
 * @param specTypeIdentifier the identifier of the SpecObject
 * 
 * @return the (first) related object or null if no related object exists
 */
function getSingleRelatedObject(specObjectIdentifier) {
	var response = sendRequestToServer(
			global.project.reqIfUrl + "/specObjects/" + specObjectIdentifier + "/relatedObjects.json",
			OPTION.ReqIFUsername,
			OPTION.ReqIFPassword,
			null
	);
	
	if (response.code == "200" && response.content) {
		var relatedObjects = JSON.parse(response.content);

		assert (relatedObjects.hasOwnProperty("values"), "Related object data did not contain any 'values' property.");
		assert (Array.isArray(relatedObjects["values"]), "Related object 'values' property is no array.");
		
		if (relatedObjects["values"].length >= 1) {
			if (relatedObjects["values"].length > 1) {
				LOGGER.warn("Object is linked to more than one SpecObject.");
			}
			return relatedObjects["values"][0];
		} else {
			LOGGER.warn("Object is not linked to any SpecObject.");
		}
	} else {
		throw new Error("Error while receiving related objects.");
	}

	return null;				
}

/**
 * Creates a cross link relation from an issue to a model element
 * 
 * @param issueUID
 * @param modelElementUID
 * 
 * @return a cross link relation for an repository snippet
 */
function createCrossLinkRelationIssue_ModelElement(issueUID, modelElementUID) {
	var crosslinkrelation = new Object();
	crosslinkrelation.relationtypeid = "com.arcway.cockpit.module.genericmodule-issuemodule2.relation.infolink.modelelementlink-genericmodule-issuemodule2--issue";
	crosslinkrelation.relationcontributions = new Object();
	crosslinkrelation.relationcontributions["com.arcway.cockpit.relation.infolink.role.moduleData"] = issueUID;
	crosslinkrelation.relationcontributions["com.arcway.cockpit.relation.infolink.role.modelElement"] = modelElementUID;
	return crosslinkrelation;
}

/**
 * Creates a cross link relation from an issue to an other module data object
 * 
 * @param issueUID
 * @param targetUID
 * 
 * @return a cross link relation for an repository snippet
 */
function createCrossLinkRelationIssue_CrossModuleDataLinks(issueUID, targetUID) {
	var crosslinkrelation = new Object();
	crosslinkrelation.relationtypeid = "com.arcway.cockpit.module.crossmodulelinks.relation.modulelib2-crossmoduledatalinks-abstract.crossmodulelink-genericmodule-issuemodule2--cross-module-issue-link";
	crosslinkrelation.relationcontributions = new Object();
	crosslinkrelation.relationcontributions["com.arcway.cockpit.module.modulelib2.relation.modulelib2-crossmoduledatalinks-abstract.role.referredItem"] = issueUID;
	crosslinkrelation.relationcontributions["com.arcway.cockpit.module.modulelib2.relation.modulelib2-crossmoduledatalinks-abstract.role.referringItem"] = targetUID;
	return crosslinkrelation;
}

/**
 * Creates a cross link relation from a requirement to a model element
 * 
 * @param requirementUID
 * @param modelElementUID
 * 
 * @return a cross link relation for an repository snippet
 */
function createCrossLinkRelationRequirement_ModelElement(requirementUID, modelElementUID) {
	var crosslinkrelation = new Object();
	crosslinkrelation.relationtypeid = "com.arcway.cockpit.module.genericmodule-requirementsmodule3.relation.infolink.modelelementlink-genericmodule-requirementsmodule3--requirement";
	crosslinkrelation.relationcontributions = new Object();
	crosslinkrelation.relationcontributions["com.arcway.cockpit.relation.infolink.role.moduleData"] = requirementUID;
	crosslinkrelation.relationcontributions["com.arcway.cockpit.relation.infolink.role.modelElement"] = modelElementUID;
	return crosslinkrelation;
}

/**
 * Parses an ReqIF identifier and returns the object type and UID
 * 
 * @param identifier the identifier containing the object type and UID
 * 
 * @return an object containing the fields "schemaObjectType", "objectUID" and "objectTypeName"
 */
function parseIdentifier(identifier) {
	if (LOGGER.isDebug()) {
		LOGGER.debug("Parsing identifier for related object. Identifier: " + JSON.stringify(identifier));
	}
	var schemaObjectType = "";
	var objectUID = "";
	var objectTypeName = "";
	
	if (identifier.startsWith(IDENTIFIER.object.requirement)) {
		if (identifier.startsWith(IDENTIFIER.object.requirement + "Folder-")) {
			schemaObjectType = "gm.requirementsmodule3.requirementSet";
			objectUID = identifier.substr(IDENTIFIER.object.requirement.length + 1 + "Folder-".length, 32);
			objectTypeName = LABEL.CockpitObjectTypeRequirementSet;
		} else if (identifier.startsWith(IDENTIFIER.object.requirement + "-")) {
			schemaObjectType = "gm.requirementsmodule3.requirement";
			objectUID = identifier.substr(IDENTIFIER.object.requirement.length + 1, 32);
			objectTypeName = LABEL.CockpitObjectTypeRequirement;
		}
	
	} else if (identifier.startsWith(IDENTIFIER.object.issue)) {
		if (identifier.startsWith(IDENTIFIER.object.issue + "Folder-")) {
			schemaObjectType = "gm.issuemodule2.issueSet";
			objectUID = identifier.substr(IDENTIFIER.object.issue.length + 1 + "Folder-".length, 32);
			objectTypeName = LABEL.CockpitObjectTypeIssueSet;
		} else if (identifier.startsWith(IDENTIFIER.object.issue + "-")) {
			schemaObjectType = "gm.issuemodule2.issue";
			objectUID = identifier.substr(IDENTIFIER.object.issue.length + 1, 32);
			objectTypeName = LABEL.CockpitObjectTypeIssue;
		}
	
	} else if (identifier.startsWith(IDENTIFIER.object.plan)) {
		if (identifier.startsWith(IDENTIFIER.object.plan + "Folder-")) {
			schemaObjectType = "folder";
			objectUID = identifier.substr(IDENTIFIER.object.plan.length + 1 + "Folder-".length, 32);
			objectTypeName = LABEL.CockpitObjectTypePlanFolder;
		} else if (identifier.startsWith(IDENTIFIER.object.plan + "-")) {
			schemaObjectType = "plan";
			objectUID = identifier.substr(IDENTIFIER.object.plan.length + 1, 32);
			objectTypeName = LABEL.CockpitObjectTypePlan;
		}
	} else if (identifier.startsWith(IDENTIFIER.object.modelElement + "-")) {
		objectTypeName = LABEL.CockpitObjectTypeModelElement;
		var subIdentifier = identifier.substr(IDENTIFIER.object.modelElement.length + 1);
		
		if (subIdentifier.startsWith(IDENTIFIER.object.modelElementFunction)) {
			schemaObjectType = "modelElement.bpre-function";
			objectUID = subIdentifier.substr(IDENTIFIER.object.modelElementFunction.length + 1, 32);
		
		} else if (subIdentifier.startsWith(IDENTIFIER.object.modelElementState)) {
			schemaObjectType = "modelElement.bpre-state";
			objectUID = subIdentifier.substr(IDENTIFIER.object.modelElementState.length + 1, 32);
		
		} else if (subIdentifier.startsWith(IDENTIFIER.object.modelElementInformation)) {
			schemaObjectType = "modelElement.bpre-information";
			objectUID = subIdentifier.substr(IDENTIFIER.object.modelElementInformation.length + 1, 32);
		}
	}
	
	if (!schemaObjectType || objectUID.length != 32) {
		LOGGER.info("No Cockpit object type or objectUID recognized in identifier. Identifier: " + identifier);
	}
	
	var parsedValues = new Object();
	parsedValues.schemaObjectType = schemaObjectType;
	parsedValues.objectUID = objectUID;
	parsedValues.objectTypeName = objectTypeName;
	
	return parsedValues;
}

/**
 * Creates a context objects and puts it in the global repository snippet
 * @param identifier
 * @returns
 */
function createContextObject(schemaTypeID, objectUID) {
	assert(schema.objecttypes.hasOwnProperty(schemaTypeID), "No schmema for schema type id '" + schemaTypeID + "' found");
	
	var schemaForType = schema.objecttypes[schemaTypeID];
	
	assert(schemaForType.hasOwnProperty("objectTypeID") && schemaForType.hasOwnProperty("uid"), "Schema information incomplete for schema type '" + schemaTypeID + "'");
	
	if (allReferredContextObjectUIDs.indexOf(objectUID) == -1) {

		var contextObject = new Object();
		contextObject.objecttypeid = schemaForType["objectTypeID"];
		contextObject.properties = new Object();
		contextObject.properties[schemaForType["uid"]] = objectUID;
		
		allReferredContextObjectUIDs.push(objectUID);
		repositorysnippet.contextobject.push(contextObject);
	}
}

/**
 * Adds a property to an object in and snippet
 * @param propertyID the (human readable) id of the property
 * @param value the value
 * @param explicitObjectProperties the property object of the snippet object
 * @param explicitObjectSchema the schema of the object type
 */
function addPropertyToSnippetObject(propertyID, value, explicitObjectProperties, explicitObjectSchema) {
	if (explicitObjectSchema.hasOwnProperty(propertyID)) {
		explicitObjectProperties[explicitObjectSchema[propertyID]] = value;
	}
}

/**
 * Returns a human redable text representing the related object
 * 
 * @param relatedSpecObject the related SpecObject as returned by the ReqIF Server
 * @param relatedSpecObjectParsedIdentifier the parsed identifier of the related SpecObject
 * 
 * @return a string with the human redable text representing the related object
 */
function getRelatedObjectAsHumanReadableText(relatedSpecObject, relatedSpecObjectParsedIdentifier) {
	var relatedObjectText = "";
	
	// Adding object type and name to description
	// - if the object type is unknown, a generic string is used
	// - if the name is unknown, the ReqIF identifier of the object is used
	if (relatedSpecObjectParsedIdentifier.hasOwnProperty("objectTypeName") && relatedSpecObjectParsedIdentifier["objectTypeName"]) {
		relatedObjectText += relatedSpecObjectParsedIdentifier["objectTypeName"];
	} else {
		relatedObjectText += LABEL.NewElementContentUnknownReqIFObject;
	}
	relatedObjectText += ": ";
	if (relatedSpecObject.hasOwnProperty("longName")) {
		relatedObjectText += relatedSpecObject["longName"];
	} else {
		relatedObjectText += relatedSpecObject["identifier"];
	}
	return relatedObjectText;
}



/* ----------------- Main --------------------*/
var allReferredContextObjectUIDs = [];

// Issue category for comments
var commentCategory = new Object();
commentCategory.humanreadableid = OPTION.IssueCategoryForComments;
commentCategory.objecttypeid = schema.objecttypes["gm.issuemodule2.issue"]["objectTypeID"];
commentCategory.categoryid = OPTION.IssueCategoryForComments;
commentCategory.sortposition = "0";
commentCategory.displayname = "";

var repositorysnippet = new Object();
repositorysnippet.explicitobject = [];
repositorysnippet.implicitobject = [];
repositorysnippet.contextobject = [];
repositorysnippet.crosslinkrelation = [];
repositorysnippet.objecttypecategory = [commentCategory];

// collect IDs of all comments, that are in the folder we will import comments to 
global.IDsOfAlreadyImportedComments = function() {
	var IDs = [];
	var issueSet = cockpit.issueDataProvider.findIssueSet(LABEL.CockpitIssueSetNameForReqIFComments);
	if (issueSet != null) {
		issueSet.getIssueChildren().toArray().forEach(function(issue) {
			IDs.push(String(issue.getID()));
		});
	}
	return IDs;
}();

// collect IDs of all new requirements, that are in the folder we will import new requirements to
global.IDsOfAlreadyImportedNewRequirements = function() {
	var IDs = [];
	var requirementSet = cockpit.requirementDataProvider.findRequirementSet(LABEL.CockpitRequirementsSetNameForReqIFNewRequirements);
	if (requirementSet != null) {
		requirementSet.getRequirementChildren().toArray().forEach(function(requirement) {
			IDs.push(String(requirement.getID()));
		});
	}
	return IDs;
}();

// collect IDs of all new issues, that are in the folder we will import new issues to
global.IDsOfAlreadyImportedNewIssues = function() {
	var IDs = [];
	var issueSet = cockpit.issueDataProvider.findIssueSet(LABEL.CockpitIssueSetNameForReqIFNewIssues);
	if (issueSet != null) {
		issueSet.getIssueChildren().toArray().forEach(function(issue) {
			IDs.push(String(issue.getID()));
		});
	}
	return IDs;
}();

// --------------- Import Comments -------------------
LOGGER.info("Retrieving comments from ReqIF server.");

(function(){
	var cockpitType = schema.objecttypes["gm.issuemodule2.issueSet"];
	
	var response = sendRequestToServer(
			global.project.reqIfUrl + "/specTypes/OT-" + IDENTIFIER.object.newComment + "/specObjects.json",
			OPTION.ReqIFUsername,
			OPTION.ReqIFPassword,
			null
	);
	
	if (response.code != "200" || !response.content) {
		throw new Error("Error while retrieving comments from ReqIF server.");
	}
	
	var content = JSON.parse(response.content);
	
	assert(content.hasOwnProperty("values") && Array.isArray(content["values"]), "Field 'values' is missing or no array. Data from Server: " + response.content);
	
	if (content["values"].length > 0) {
		
		// IssueSet for comments
		var implicitobject = new Object();
		implicitobject.repositoryobjectsample = new Object();
		implicitobject.repositoryobjectsample.objecttypeid = cockpitType["objectTypeID"];
		implicitobject.repositoryobjectsample.category = null;
		implicitobject.repositoryobjectsample.properties = new Object();
		implicitobject.repositoryobjectsample.properties[cockpitType["uid"]] = cockpit.UUIDGenerator.getUniqueID() + "";
		implicitobject.repositoryobjectsample.properties[cockpitType["name"]] = LABEL.CockpitIssueSetNameForReqIFComments;
		implicitobject.parentobjectreference = null;
		
		repositorysnippet.implicitobject.push(implicitobject);		
		
		// Reference to issue set for issues
		var repositoryobjectreference = new Object();
		repositoryobjectreference.objecttypeid = implicitobject.repositoryobjectsample.objecttypeid;
		repositoryobjectreference.objectid = implicitobject.repositoryobjectsample.properties[cockpitType["uid"]];
		
		// Create issues from ReqIF comments
		content["values"].forEach(function(comment) {
			var objects = parseComment(comment, repositoryobjectreference);
			repositorysnippet.explicitobject.addAll(objects.explicitobject);
			repositorysnippet.crosslinkrelation.addAll(objects.crosslinkrelation);
		});
	} else {
		LOGGER.info("No comments found on ReqIF server.");
	}
})();

/**
 * Builds a URL that points to a specific spec object in the current project in the iLAH
 * @param identifier identifier of the object
 * @return URL as sSring 
 */
function buildURLToObjektInILAH(identifier) {
	return global.urlbuilder.getILAHURL("projects/" + global.project.reqIfIdentifier + "/Hierarchy-" + identifier);
}

// --------------- Import Requirements -------------------
LOGGER.info("Retrieving new requirements from ReqIF server.");

(function(){
	var cockpitType = schema.objecttypes["gm.requirementsmodule3.requirementSet"];
	
	var response = sendRequestToServer(
			global.project.reqIfUrl + "/specTypes/OT-" + IDENTIFIER.object.newRequirement + "/specObjects.json",
			OPTION.ReqIFUsername,
			OPTION.ReqIFPassword,
			null
	);
	
	if (response.code != "200" || !response.content) {
		throw new Error("Error while retrieving new requirements from ReqIF server. Received " + contentAsJSON);
	}
	
	var content = JSON.parse(response.content);
	
	assert(content.hasOwnProperty("values") && Array.isArray(content["values"]), "Field 'values' is missing or no array. Data from Server: " + response.content);
	
	if (content["values"].length > 0) {
		
		// Requirement set for new requirements
		var implicitobject = new Object();
		implicitobject.repositoryobjectsample = new Object();
		implicitobject.repositoryobjectsample.objecttypeid = cockpitType["objectTypeID"];
		implicitobject.repositoryobjectsample.category = null;
		implicitobject.repositoryobjectsample.properties = new Object();
		implicitobject.repositoryobjectsample.properties[cockpitType["uid"]] = String(cockpit.UUIDGenerator.getUniqueID());
		implicitobject.repositoryobjectsample.properties[cockpitType["name"]] = LABEL.CockpitRequirementsSetNameForReqIFNewRequirements;
		implicitobject.parentobjectreference = null;
		
		repositorysnippet.implicitobject.push(implicitobject);		
		
		// Reference to requirement set for requirements
		var repositoryobjectreference = new Object();
		repositoryobjectreference.objecttypeid = implicitobject.repositoryobjectsample.objecttypeid;
		repositoryobjectreference.objectid = implicitobject.repositoryobjectsample.properties[cockpitType["uid"]];
		
		// Create requirements from ReqIF new requirements
		content["values"].forEach(function(newRequirement) {
			var objects = parseNewRequirement(newRequirement, repositoryobjectreference);
			repositorysnippet.explicitobject.addAll(objects.explicitobject);
			repositorysnippet.crosslinkrelation.addAll(objects.crosslinkrelation);
		});
	} else {
		LOGGER.info("No new requirements found on ReqIF server.");
	}
})();


// --------------- Import Issues -------------------
LOGGER.info("Retrieving new issues from ReqIF server.");

(function(){
	var cockpitType = schema.objecttypes["gm.issuemodule2.issueSet"];
	
	var response = sendRequestToServer(
			global.project.reqIfUrl + "/specTypes/OT-" +  IDENTIFIER.object.newIssue + "/specObjects.json",
			OPTION.ReqIFUsername,
			OPTION.ReqIFPassword,
			null
	);
	
	if (response.code != "200" || !response.content) {
		throw new Error("Error while retrieving new issues from ReqIF server. Received " + contentAsJSON);
	}
	
	var content = JSON.parse(response.content);
	
	assert(content.hasOwnProperty("values") && Array.isArray(content["values"]), "Field 'values' is missing or no array. Data from Server: " + response.content);
	
	if (content["values"].length > 0) {
		
		// Issue set for new issues
		var implicitobject = new Object();
		implicitobject.repositoryobjectsample = new Object();
		implicitobject.repositoryobjectsample.objecttypeid = cockpitType["objectTypeID"];
		implicitobject.repositoryobjectsample.category = null;
		implicitobject.repositoryobjectsample.properties = new Object();
		implicitobject.repositoryobjectsample.properties[cockpitType["uid"]] = String(cockpit.UUIDGenerator.getUniqueID());
		implicitobject.repositoryobjectsample.properties[cockpitType["name"]] = LABEL.CockpitIssueSetNameForReqIFNewIssues;
		implicitobject.parentobjectreference = null;
		addPropertyToSnippetObject(OPTION.CustomPropertyExportVisibility, "true", implicitobject.repositoryobjectsample.properties, cockpitType);
		
		repositorysnippet.implicitobject.push(implicitobject);		
		
		// Reference to issue set for issues
		var repositoryobjectreference = new Object();
		repositoryobjectreference.objecttypeid = implicitobject.repositoryobjectsample.objecttypeid;
		repositoryobjectreference.objectid = implicitobject.repositoryobjectsample.properties[cockpitType["uid"]];
		
		// Create issues from ReqIF new issues
		content["values"].forEach(function(newIssue) {
			var objects = parseNewIssue(newIssue, repositoryobjectreference);
			repositorysnippet.explicitobject.addAll(objects.explicitobject);
			repositorysnippet.crosslinkrelation.addAll(objects.crosslinkrelation);
		});
	} else {
		LOGGER.info("No new issues found on ReqIF server.");
	}
})();
	
// root object for elements with parent reference null
var OBJECTREFERENCE_REPOSITORYROOTOBJECT = new Object(); 
OBJECTREFERENCE_REPOSITORYROOTOBJECT.objecttypeid = "com.arcway.cockpit.object.data.project";
OBJECTREFERENCE_REPOSITORYROOTOBJECT.objectid = String(global.project.project.getUniqueIdentifier());

if (LOGGER.isDebug()) {
	LOGGER.debug("Final repository snippet: " + JSON.stringify(repositorysnippet));
}

global.importing = new Object();
global.importing.successful = undefined;

if (repositorysnippet.explicitobject.length > 0) {
	global.importing.successful = false;
	LOGGER.debug("Creating import job to import " + repositorysnippet.explicitobject.length + " objects.");
	var importJob = cockpit.ImportJobFactory.createImportJob();
	
	LOGGER.debug("Setting repository snippet in import job.");
	importJob.setRepositorySnippet(JSON.stringify(repositorysnippet));
	
	var targetObjectReferenceMap = new Object();
	targetObjectReferenceMap.entries = [];
	var mapEntry = new Object();
	mapEntry.key = schema.objecttypes["gm.issuemodule2.issueSet"]["objectTypeID"];
	mapEntry.value = OBJECTREFERENCE_REPOSITORYROOTOBJECT;
	targetObjectReferenceMap.entries.push(mapEntry);
	
	mapEntry = new Object();
	mapEntry.key = schema.objecttypes["gm.requirementsmodule3.requirementSet"]["objectTypeID"];
	mapEntry.value = OBJECTREFERENCE_REPOSITORYROOTOBJECT;
	targetObjectReferenceMap.entries.push(mapEntry);

	// Matching object type categories
	var objectTypesToBeConsideredForCategoryMatching = new Object(); 
	objectTypesToBeConsideredForCategoryMatching.repositoryobjecttypeid = [];
	objectTypesToBeConsideredForCategoryMatching.repositoryobjecttypeid.push(schema.objecttypes["gm.issuemodule2.issue"]["objectTypeID"]);
	objectTypesToBeConsideredForCategoryMatching.repositoryobjecttypeid.push(schema.objecttypes["gm.requirementsmodule3.requirement"]["objectTypeID"]);
	
	importJob.addImportJobEditor("AIJEObjectTypeCategoryRecogniser", JSON.stringify(objectTypesToBeConsideredForCategoryMatching), JSON.stringify("false"), JSON.stringify("true"));
	
	objectTypesToBeConsideredForCategoryMatching = new Object(); 
	objectTypesToBeConsideredForCategoryMatching.repositoryobjecttypeid = [];
	objectTypesToBeConsideredForCategoryMatching.repositoryobjecttypeid.push(schema.objecttypes["gm.issuemodule2.issue"]["objectTypeID"]);
	objectTypesToBeConsideredForCategoryMatching.repositoryobjecttypeid.push(schema.objecttypes["gm.requirementsmodule3.requirement"]["objectTypeID"]);
	importJob.addImportJobEditor("AIJEObjectTypeCategoryToObjectAssigner", JSON.stringify(objectTypesToBeConsideredForCategoryMatching));
	
	// original from Java: importJobEditors.add( new AIJETargetChooserForRootObjects( map_objectType_rootObject ) );
	importJob.addImportJobEditor("AIJETargetChooserForRootObjects", JSON.stringify(targetObjectReferenceMap));
	
	// original from Java: importJobEditors.add( new AIJEObjectRecogniserByName(COTIDsSection.OBJECT_TYPE_ID, COTIDsSection.PROPERTY_TYPE_ID_NAME));
	importJob.addImportJobEditor("AIJEObjectRecogniserByName", JSON.stringify(schema.objecttypes["gm.issuemodule2.issueSet"]["objectTypeID"]), JSON.stringify(schema.objecttypes["gm.issuemodule2.issueSet"]["name"]));
	importJob.addImportJobEditor("AIJEObjectRecogniserByName", JSON.stringify(schema.objecttypes["gm.requirementsmodule3.requirementSet"]["objectTypeID"]), JSON.stringify(schema.objecttypes["gm.requirementsmodule3.requirementSet"]["name"]));
	
	var objectTypesToBeConsideredForObjectMatching = new Object(); 
	objectTypesToBeConsideredForObjectMatching.repositoryobjecttypeid = [];
	objectTypesToBeConsideredForObjectMatching.repositoryobjecttypeid.push(schema.objecttypes["gm.issuemodule2.issue"]["objectTypeID"]);
	objectTypesToBeConsideredForObjectMatching.repositoryobjecttypeid.push(schema.objecttypes["gm.requirementsmodule3.requirement"]["objectTypeID"]);
	objectTypesToBeConsideredForObjectMatching.repositoryobjecttypeid.push(schema.objecttypes["modelElement"]["objectTypeID"]);
	
	importJob.addImportJobEditor("AIJEObjectRecogniserByUID", JSON.stringify(objectTypesToBeConsideredForObjectMatching), JSON.stringify("false"), JSON.stringify("true"));
	
	// original from Java: importJobEditors.add( new AIJEObjectImportModeSelector(AIJEObjectImportModeSelector.ExecutionMode.EXECUTIONMODE_MERGE_IMPLICIT_MATCHING));
	importJob.addImportJobEditor("AIJEObjectImportModeSelector", JSON.stringify("EXECUTIONMODE_MERGE_IMPLICIT_MATCHING"));
	
	if (OPTION.AlwaysShowMerge === true) {
		importJob.addImportJobEditor("MIJEImportDialog", JSON.stringify("true"));
	}
	
	LOGGER.info("Starting import job.");
	importJob.executeImportJob();
	
	var importException = importJob.getImportExportException();
	
	if (importException === null) {
		LOGGER.info("Import job done.");
		return true;
	} else {
		LOGGER.error("Import job failed due to following problems:");
		importJob.getImportMessages().getMessages().asJavaCollection().toArray().forEach(function(errorMessage) {
			LOGGER.error(errorMessage.toString());
		});
		
		throw importException;
	}
} else {
	LOGGER.info("Not creating import job as no objects to import were found.");
}
return false;
}