/*
 * Export from Cockpit to ReqIF Server.
 * 
 * This script provides a function that
 * - creates the XML tree for all objects that shall be exproted
 * - wraps the XML and all exported files in a ZipStream and sends it to the server 
 */

/**
 * Starts the export.
 * 
 * @param increaseVersionCountByOne if the version counter should be one higher than the actual project version being exported
 * 
 * @throws Error on unrecoverable error
 */
function startExportToReqIF(increaseVersionCountByOne) {

/**
 * Constructor for a new JS XMLDocument object
 * 
 * @param name the name of the root node
 * @param disableEscaping wether to disable escaping of special characters (&, <, >, ...) or not
 * 
 * @return {XMLDocument}the new xml document containing two properties for the java Document ("doc") and the root Java Node ("rootNode") 
 */
function XMLDocument(name, disableEscaping) {
	var docFactory = javax.xml.parsers.DocumentBuilderFactory.newInstance();
	var docBuilder = docFactory.newDocumentBuilder();
	this.doc = docBuilder.newDocument();
	this.doc.setXmlStandalone(true);
	if (disableEscaping) {
		this.doc.appendChild(this.doc.createProcessingInstruction(javax.xml.transform.stream.StreamResult.PI_DISABLE_OUTPUT_ESCAPING,""));
	}
	this.rootNode = new XMLNode(this.doc, this.doc, name);
}

/**
 * Constructor for a new JS XMLNode object
 * 
 * @param parent the parent node or document
 * @param doc the document to create the node in
 * @param name the name of the node
 * 
 * @return {XMLNode} the new JS XML Node
 */
function XMLNode(parent, doc, name) {
	this.elem = {
		_javaObj : doc.createElement(name),
		_javaDocument : doc
	};
	parent.appendChild(this.elem._javaObj);
}

/**
 * Creates a new child XMLNode object
 * 
 * @param name the name of the new child node
 */
XMLNode.prototype.createChild = function (name) {
	return new XMLNode(this.elem._javaObj, this.elem._javaDocument, name);
};

/**
 * Writes text in an XMLNode
 */
XMLNode.prototype.createText = function (content) {
	var textNode = this.elem._javaDocument.createTextNode(content);
	this.elem._javaObj.appendChild(textNode);
};

/**
 * Creates attributes in an XMLNode
 */
XMLNode.prototype.createAttribute = function (name, content) {
	var attrNode = this.elem._javaDocument.createAttribute(name);
	attrNode.setValue(content);
	this.elem._javaObj.setAttributeNode(attrNode);
};

/**
 * Writes an XML Document to an OutputStream as a String
 * 
 * @param document the Java XML Document
 * @param outputStream a Java OutputStream
 * 
 */
function getXMLFromDocument(document, outputStream) {
	var transformerFactory = javax.xml.transform.TransformerFactory.newInstance();
	var transformer = transformerFactory.newTransformer();
	transformer.setOutputProperty(javax.xml.transform.OutputKeys.INDENT, "yes");
	transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");
	transformer.setOutputProperty("encoding", "UTF-8");
	var source = new javax.xml.transform.dom.DOMSource(document);
	transformer.transform(source, new javax.xml.transform.stream.StreamResult(outputStream));
}

/**
 * Zips the contents,
 * 
 * @param outputStream Java OutputStream
 * @param xmlDocument Java XML Document
 * @param files array of objects that have the fields
 * <ul><li>fileName</li><li>pathInZip</li><li>pathOnDisk</li></ul>
 */
function zipContents(outputStream, xmlDocument, files) {
	var zipStream = new java.util.zip.ZipOutputStream(outputStream);
	
	zipStream.putNextEntry(new java.util.zip.ZipEntry("CoPilot.xml"));
	getXMLFromDocument(xmlDocument, zipStream);
	zipStream.closeEntry();
	
	var bytes = java.lang.reflect.Array.newInstance(java.lang.Byte.TYPE, 16384);
	var nrBytesRead = -1;
	
	files.forEach(function(file) {
		var zipEntry = new java.util.zip.ZipEntry(new java.io.File(file.pathInZip, file.fileName));
		zipStream.putNextEntry(zipEntry);
		var fileInputStream = new java.io.FileInputStream(new java.io.File(file.pathOnDisk, file.fileName));
		
		while ((nrBytesRead = fileInputStream.read(bytes)) != -1) {
			zipStream.write(bytes, 0, nrBytesRead);
		}
		
		fileInputStream.close();
		zipStream.closeEntry();
	});

	zipStream.close();
}

/**
 * Writes the spec types for all custom properties of an object type
 * 
 * @param xmlNode the XMLNode to write to
 * @param prefix the prefix for the identifiers of the spec types
 * @param objectTypeID the object type ID whose custom properties should be written
 * @param categoryID the category ID for which the custom properties should be written (null for default category)
 */
function writeCustomPropertyTypes(xmlNode, prefix, objectTypeID, categoryID) {
	var categoryRifID;
	var customprops;
	if ( categoryID === null ) {
		categoryRifID = "";
		customprops = cockpit.dataModelProvider.getCustomPropertyIdsForObjectType( objectTypeID );
	} else {
		categoryRifID = "-" + categoryID;
		customprops = cockpit.dataModelProvider.getCustomPropertiesForObjectTypeCategory( objectTypeID, categoryID );
	}

	customprops.toArray().forEach(function(customPropertyID) {
		if (!exportCustomProperty(customPropertyID)) {
			return;
		}
		var displayName = cockpit.dataModelProvider.getDisplaynameOfProperty( objectTypeID, customPropertyID );
		var typeID = cockpit.dataModelProvider.getDatatypeIDOfProperty( objectTypeID, customPropertyID );
		
		// Attribute Type
		var attrDef = null;		
		if (typeID.equals("url") || typeID.equals("string") || typeID.equals("text")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-DEFINITION-STRING");
		} else if (typeID.equals("boolean")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-DEFINITION-BOOLEAN");
		} else if (typeID.equals("date")) {
			// There currently is no way to have differently formatted datatype DATE dates so treat them as type STRING
			attrDef = xmlNode.createChild("ATTRIBUTE-DEFINITION-STRING");
//			attrDef = xmlNode.createChild("ATTRIBUTE-DEFINITION-DATE");
		} else if (typeID.equals("integer")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-DEFINITION-INTEGER");
		} else if (typeID.equals("double")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-DEFINITION-REAL");
		} else if (typeID.equals("enumeration_string_single") || typeID.equals("enumeration_string_multiple")
			|| typeID.equals("enumeration_integer_single") || typeID.equals("enumeration_integer_multiple")) {
			
			attrDef = xmlNode.createChild("ATTRIBUTE-DEFINITION-ENUMERATION");
		} else if (typeID.equals("file")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-DEFINITION-XHTML");
		}
		
		assert(attrDef != null, "Unknown custom property datatype '" + typeID + "'.");
		
		// common attributes for all types
		attrDef.createAttribute("IDENTIFIER", "AD-" + prefix + categoryRifID + "-" + customPropertyID);
		attrDef.createAttribute("LONG-NAME", displayName);
		attrDef.createAttribute("LAST-CHANGE", templateTime);
		
		// "Multi Valued" for enumeration tyes
		if (typeID.equals("enumeration_string_single") || typeID.equals("enumeration_string_multiple")
				|| typeID.equals("enumeration_integer_single") || typeID.equals("enumeration_integer_multiple")) {
			
			if (typeID.equals("enumeration_string_single") || typeID.equals("enumeration_integer_single")) {
				attrDef.createAttribute("MULTI-VALUED", "false");
			} else {
				attrDef.createAttribute("MULTI-VALUED", "true");
			}
		}
		
		// Attribute Data Type
		if (typeID.equals("date")) {
			// There currently is no way to have differently formatted datatype DATE dates so treat them as type STRING
			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
//			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-DATE-REF").createText("DT-ISODate");
		} else if (typeID.equals("string")) {
			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
		} else if (typeID.equals("text")) {
			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
		} else if (typeID.equals("url")) {
			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
		} else if (typeID.equals("boolean")) {
			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-BOOLEAN-REF").createText("DT-Boolean");
		} else if (typeID.equals("integer")) {
			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-INTEGER-REF").createText("DT-Integer");
		} else if (typeID.equals("double")) {
			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-REAL-REF").createText("DT-Real");
		} else if (typeID.equals("enumeration_string_single") || typeID.equals("enumeration_string_multiple")
				|| typeID.equals("enumeration_integer_single") || typeID.equals("enumeration_integer_multiple") ) {
			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-ENUMERATION-REF").createText("DT-" + prefix + "-" + customPropertyID);
		} else if (typeID.equals("file")) {
			attrDef.createChild("TYPE").createChild("DATATYPE-DEFINITION-XHTML-REF").createText("DT-FormattedText");
		}
	});
}

/**
 * Writes an enum value
 * 
 * @param xmlNode the XMLNode to write to
 * @param longName the long name of the enum value
 * @param identifier the identifier of the enum value
 * @param lastChange the time of the last change of the enum value
 * @param key the key of the enum value
 */
function writeEnumValue(xmlNode, longName, identifier, lastChange, key) {
	var enumValue = xmlNode.createChild("ENUM-VALUE");
	enumValue.createAttribute("IDENTIFIER", identifier);
	enumValue.createAttribute("LONG-NAME", longName);
	enumValue.createAttribute("LAST-CHANGE", lastChange);
	
	var embeddeValue = enumValue.createChild("PROPERTIES").createChild("EMBEDDED-VALUE");
	embeddeValue.createAttribute("KEY", key);
	embeddeValue.createAttribute("OTHER-CONTENT", "");
}

/**
 * Writes all enum definitions for an object type
 * 
 * @param xmlNode the XMLNode to write to
 * @param objectTypeID the object type ID
 * @param prefix the prefix for the identifiers
 */
function writeEnumCustomPropertyDatatypeDefs(xmlNode, objectTypeID, prefix) {

	var customprops = cockpit.dataModelProvider.getCustomPropertyIdsForObjectType(objectTypeID);
	customprops.toArray().forEach(function(customPropertyID) {
		if (cockpit.dataModelProvider.isEnumerationCustomProperty(objectTypeID, customPropertyID)) {
			var datatype = xmlNode.createChild("DATATYPE-DEFINITION-ENUMERATION");
			datatype.createAttribute("IDENTIFIER", "DT-" + prefix + "-" + customPropertyID);
			datatype.createAttribute("LONG-NAME", cockpit.dataModelProvider.getDisplaynameOfProperty(objectTypeID, customPropertyID));
			datatype.createAttribute("DESC", "Enumerated Values for the type " + objectTypeID + " " + customPropertyID);
			datatype.createAttribute("LAST-CHANGE", templateTime);
			
			var valuesCount = 0;
			var	values = datatype.createChild("SPECIFIED-VALUES");
			cockpit.dataModelProvider.getAllowedValuesOfCustomProperty(objectTypeID, customPropertyID).toArray().forEach(function(val) {
				writeEnumValue(values, val, "V-" + prefix + "-" + customPropertyID + "-" + valuesCount, templateTime, valuesCount);
				valuesCount++;
			});
		}
	});
}

/**
 * Writes all custom property values for an item 
 * 
 * @param xmlNode the XMLNode to write to
 * @param prefix the prefix for the identifiers
 * @param item the item
 */
function writeCustomProperties(xmlNode, prefix, item) {
	var itemID = item.getUniqueIdentifier();
	var categoryRifID;
	var customProperties;
	if (item.hasDefaultCategory()) {
		categoryRifID = "";
		customProperties = cockpit.dataModelProvider.getCustomPropertyIdsForObjectType(item.getTypeID());
	} else {
		categoryRifID = "-" + item.getCategoryID();
		customProperties = cockpit.dataModelProvider.getCustomPropertiesForObjectTypeCategory(item.getTypeID(), item.getCategoryID());
	}
	
	customProperties.toArray().forEach(function(customPropertyID) {
		if (!exportCustomProperty(customPropertyID) || !item.hasPropertyValue(customPropertyID)) {
			return;
		}
		var typeID = cockpit.dataModelProvider.getDatatypeIDOfProperty(item.getTypeID(), customPropertyID);
		
		var attrDef = null;
		if (typeID.equals("url") || typeID.equals("string") || typeID.equals("text")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-VALUE-STRING");
		} else if (typeID.equals("boolean")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-VALUE-BOOLEAN");
		} else if (typeID.equals("date")) {
			// There currently is no way to have differently formatted datatype DATE dates so treat them as type STRING
			attrDef = xmlNode.createChild("ATTRIBUTE-VALUE-STRING");
//			attrDef = xmlNode.createChild("ATTRIBUTE-VALUE-DATE");
		} else if (typeID.equals("integer")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-VALUE-INTEGER");
		} else if (typeID.equals("double")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-VALUE-REAL");
		} else if (typeID.equals("enumeration_string_single") || typeID.equals("enumeration_integer_single") 
				|| typeID.equals("enumeration_string_multiple") || typeID.equals("enumeration_integer_multiple")) {
			attrDef = xmlNode.createChild("ATTRIBUTE-VALUE-ENUMERATION");
		} else if (typeID.equals("file")) {
			if (item.hasPropertyValue(customPropertyID)) {
				attrDef = xmlNode.createChild("ATTRIBUTE-VALUE-XHTML");
			} else {
				return;
			}
		}
		
		assert(attrDef != null, "Unknown custom property datatype '" + typeID + "'.");
		
		if (typeof attrDef === "undefined" || !attrDef) {
			LOGGER.debug("TypeID '" + typeID + "' led to attribute definition '" + attrDef + "'");
			return;
		}
		
		var definition = attrDef.createChild("DEFINITION");
		
		var attributeDefRef = "AD-" + prefix + categoryRifID + "-" + customPropertyID;

		if (typeID.equals("url") || typeID.equals("string") || typeID.equals("text")) {
			definition.createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText(attributeDefRef);
			
			var theText;
			if (typeID.equals("text")) {
				theText = "";
				item.getPropertyAsStringArray(customPropertyID).forEach(function(line) {
					theText += line;
				});
			} else {
				theText = item.getPropertyAsString(customPropertyID);
			}
			
			attrDef.createAttribute("THE-VALUE", theText);

		} else if (typeID.equals("boolean")) {
			definition.createChild("ATTRIBUTE-DEFINITION-BOOLEAN-REF").createText(attributeDefRef);
			
			var javaBool = item.getPropertyAsJavaObject(customPropertyID);
			
			attrDef.createAttribute("THE-VALUE", String(javaBool.toString()));
			
		} else if (typeID.equals("date")) {
			// There currently is no way to have differently formatted datatype DATE dates so treat them as type STRING
			definition.createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText(attributeDefRef);
			attrDef.createAttribute("THE-VALUE", item.getPropertyAsString(customPropertyID));
						
//			definition.createChild("ATTRIBUTE-DEFINITION-DATE-REF").createText(attributeDefRef);
//						
//			var theDateValue = item.getDateTimeProperty_MilliSecondsSince_01_01_1970(customPropertyID);
//			
//			attrDef.createAttribute("THE-VALUE", getISODateTimeWithSeconds(theDateValue));
			
		} else if (typeID.equals("integer") || typeID.equals("double")) {
			if (typeID.equals("integer")) {
				definition.createChild("ATTRIBUTE-DEFINITION-INTEGER-REF").createText(attributeDefRef);
			} else if (typeID.equals("double")) {
				definition.createChild("ATTRIBUTE-DEFINITION-REAL-REF").createText(attributeDefRef);
			}
			
			attrDef.createAttribute("THE-VALUE", String(item.getPropertyAsJavaObject(customPropertyID).toString()));
		
		} else if (typeID.equals("enumeration_string_single") || typeID.equals("enumeration_integer_single") 
				|| typeID.equals("enumeration_string_multiple") || typeID.equals("enumeration_integer_multiple")) {
		
			definition.createChild("ATTRIBUTE-DEFINITION-ENUMERATION-REF").createText(attributeDefRef);
			
			var values = attrDef.createChild("VALUES");

			if (item.hasPropertyValue(customPropertyID)) {
				var allValues = cockpit.dataModelProvider.getAllowedValuesOfCustomProperty(item.getTypeID(), customPropertyID);
				if (typeID.equals("enumeration_string_single") || typeID.equals("enumeration_integer_single")) {
					var val = item.getPropertyAsString(customPropertyID);
					values.createChild("ENUM-VALUE-REF").createText("V-" + prefix + "-" + customPropertyID + "-" + allValues.indexOf(val));
				} else {
					var propertyAsStringList = item.getPropertyAsStringList(customPropertyID);
					propertyAsStringList.toArray().forEach(function(val) {
						values.createChild("ENUM-VALUE-REF").createText("V-" + prefix + "-" + customPropertyID + "-" + allValues.indexOf(val));
					});
				}
			}
		} else if (typeID.equals("file")) {
			definition.createChild("ATTRIBUTE-DEFINITION-XHTML-REF").createText(attributeDefRef);
			var xhtmlContentDiv = attrDef.createChild("THE-VALUE").createChild("xhtml:div");
			var file = getPath(item.getFileProperty(customPropertyID));
			if (file != null) {
				if (includeFileInZip(file.getName(), cockpit.tempImageDirectory, global.zip.folderInZipForFiles)) {
					if (isImage(file.getName())) {
						var xhtmlObject = xhtmlContentDiv.createChild("xhtml:object");
						var mimeType = mimeTypeForExtension(file.getName());
						if (mimeType == null) {
							mimeType = "application/octet-stream";
						}
						xhtmlObject.createAttribute("name", file.getName());
						xhtmlObject.createAttribute("type", mimeType);
						xhtmlObject.createAttribute("data", file.toString().replace(java.io.File.separator, "/"));
					} else {
						var xhtmlA = xhtmlContentDiv.createChild("xhtml:a");
						xhtmlA.createAttribute("title", file.getName());
						xhtmlA.createAttribute("href", "/reqif/services/projects/" + 
								global.project.reqIfIdentifier + "/files/" + file.toString().replace(java.io.File.separator, "/"));
						xhtmlA.createText(file.getName());
					}
				} else {
					xhtmlContentDiv.createChild("File not included due to exceeded file size sum.");
				}
			} else {
				LOGGER.error("File for object '" + itemID + "' and custom property id '" + customPropertyID + "' not found");
				xhtmlContentDiv.createText("File not found!");
			}
		}
	});
}

/**
 * Writes a SpecObject for a requirement
 * 
 * @param xmlNode
 * @param req
 * 
 * @return the ReqIF Identifier of the written object
 */
function writeSpecObjectRequirement (xmlNode, req) {
	var identifier = composeGUID(IDENTIFIER.object.requirement + "-" + req.getUniqueIdentifier());
	var date = getLastChangeValue(req);
	
	var categoryIDRif;
	if (req.hasDefaultCategory()) {
		categoryIDRif = "";
	} else {
		categoryIDRif = "-" + req.getCategoryID();
	}
	
	var refSuffix = IDENTIFIER.object.requirement + categoryIDRif;
	
	var specObject = xmlNode.createChild("SPEC-OBJECT");
		
	specObject.createAttribute("IDENTIFIER", identifier);
	specObject.createAttribute("LONG-NAME", req.getName());
	specObject.createAttribute("LAST-CHANGE", date);
	specObject.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-" + refSuffix);
	
	var values = specObject.createChild("VALUES");
	
	var attrID = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrID.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectID-" + refSuffix);
	attrID.createAttribute("THE-VALUE", req.getID());

	var attrHeading = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrHeading.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-" + refSuffix);
	attrHeading.createAttribute("THE-VALUE", req.getName());
	
	var attrDescription = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrDescription.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectText-" + refSuffix);
	
	var description = "";
	req.getDescription().toArray().forEach(function(line) {
		description += line;
	});
	attrDescription.createAttribute("THE-VALUE", description);
	
	var attrStatus = values.createChild("ATTRIBUTE-VALUE-ENUMERATION");
	attrStatus.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-ENUMERATION-REF").createText("AD-Status-" + refSuffix);

	var attrStatusValues = attrStatus.createChild("VALUES");
	if (req.getState().length() > 0 ) {
		attrStatusValues.createChild("ENUM-VALUE-REF")
				.createText("V-" + IDENTIFIER.object.requirement + "-status-" + global.requirement.allStatusValues.indexOf(req.getState()));
	}
	
	var attrPriority = values.createChild("ATTRIBUTE-VALUE-ENUMERATION");
	attrPriority.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-ENUMERATION-REF").createText("AD-Priority-" + refSuffix);

	var attrattrPriorityValues = attrPriority.createChild("VALUES");
	if (req.getPriority().length() > 0 ) {
		attrattrPriorityValues.createChild("ENUM-VALUE-REF")
		.createText("V-" + IDENTIFIER.object.requirement + "-prio-" + global.requirement.allPrioValues.indexOf(req.getPriority()));
	}
	
	// Create an ATTRIBUTE-VALUE per Custom Property
	writeCustomProperties(values, IDENTIFIER.object.requirement, req);
	
	writeReqIFUserAttributeValue(values, refSuffix, {"cockpitObject" : req});
	
	return identifier;
}

/**
 * Writes a SpecObject for a model element
 * 
 * @param xmlNode
 * @param modelElement
 * 
 * @return the ReqIF Identifier of the written object
 */
function writeSpecObjectModelElement(xmlNode, modelElement) {
	var identifier = getRifIdentifierForModelElement(modelElement);
	var date = getLastChangeValue(modelElement);
	
	var categoryIDRif;
	if (modelElement.hasDefaultCategory()) {
		categoryIDRif = "";
	} else {
		categoryIDRif = "-" + modelElement.getCategoryID();
	}
	
	var refSuffix = IDENTIFIER.object.modelElement + categoryIDRif;
	
	var specObject = xmlNode.createChild("SPEC-OBJECT");
	
	specObject.createAttribute("IDENTIFIER", identifier);
	specObject.createAttribute("LONG-NAME", modelElement.getName());
	specObject.createAttribute("LAST-CHANGE", date);
	specObject.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-" + refSuffix);
	
	var values = specObject.createChild("VALUES");
	
	var attrHeading = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrHeading.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-" + refSuffix);
	attrHeading.createAttribute("THE-VALUE", modelElement.getName());
	
	var attrDescription = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrDescription.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectText-" + refSuffix);

	var description = "";
	modelElement.getDescription().toArray().forEach(function(line) {
		description += (line);
	});
	attrDescription.createAttribute("THE-VALUE", description);

	var attrModelElementType = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrModelElementType.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-Type-" + refSuffix);
	attrModelElementType.createAttribute("THE-VALUE", modelElement.getElementTypeName());

	// Custom Properties
	writeCustomProperties(values, IDENTIFIER.object.modelElement, modelElement);
	
	writeReqIFUserAttributeValue(values, refSuffix, {"cockpitObject" : modelElement});
	
	return identifier;
}


/**
 * Write a SpecObject for an issue
 * 
 * @param xmlNode
 * @param issue
 * 
 * @return the ReqIF Identifier of the written object
 */
function writeSpecObjectIssue(xmlNode, issue) {
	var identifier = composeGUID(IDENTIFIER.object.issue + "-" + issue.getUniqueIdentifier());
	var date = getLastChangeValue(issue);
	
	var categoryIDRif;
	if (issue.hasDefaultCategory()) {
		categoryIDRif = "";
	} else {
		categoryIDRif = "-" + issue.getCategoryID();
	}
	
	var refSuffix = IDENTIFIER.object.issue + categoryIDRif;
	
	var specObject = xmlNode.createChild("SPEC-OBJECT");
	
	specObject.createAttribute("IDENTIFIER", identifier);
	specObject.createAttribute("LONG-NAME", issue.getTitle());
	specObject.createAttribute("LAST-CHANGE", date);
	specObject.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-" + refSuffix);
	
	var values = specObject.createChild("VALUES");

	var attrID = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrID.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectID-" + refSuffix);
	attrID.createAttribute("THE-VALUE", issue.getID());
	
	var attrHeading = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrHeading.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-" + refSuffix);
	attrHeading.createAttribute("THE-VALUE", issue.getTitle());
	
	var attrDescription = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrDescription.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectText-" + refSuffix);
	
	var description = "";
	issue.getDescription().toArray().forEach(function(line) {
		description += line;
	});
	attrDescription.createAttribute("THE-VALUE", description);
	
	var attrStatus = values.createChild("ATTRIBUTE-VALUE-ENUMERATION");
	attrStatus.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-ENUMERATION-REF").createText("AD-Status-" + refSuffix);

	var attrStatusValues = attrStatus.createChild("VALUES");
	if (issue.getStatus().length() > 0 ) {
		attrStatusValues.createChild("ENUM-VALUE-REF")
				.createText("V-" + IDENTIFIER.object.issue + "-status-" + global.issue.allStatusValues.indexOf(issue.getStatus()));
	}
	
	var attrPriority = values.createChild("ATTRIBUTE-VALUE-ENUMERATION");
	attrPriority.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-ENUMERATION-REF").createText("AD-Priority-" + refSuffix);

	var attrattrPriorityValues = attrPriority.createChild("VALUES");
	if (issue.getPriority().length() > 0) {
		attrattrPriorityValues.createChild("ENUM-VALUE-REF")
				.createText("V-" + IDENTIFIER.object.issue + "-prio-" + global.issue.allPrioValues.indexOf(issue.getPriority()));
	}
	
	var attrResponsible = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrResponsible.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-Responsible-" + refSuffix);
	attrResponsible.createAttribute("THE-VALUE", issue.getResponsible());

	// There currently is no way to have differently formatted datatype DATE dates so treat them as type STRING
	var attrDueDate = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrDueDate.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-DueDate-" + refSuffix);
	attrDueDate.createAttribute("THE-VALUE", issue.getDueDate());
	
//	var dueDateValue = issue.getDueDate_MilliSecondsSince_01_01_1970();
//	if (dueDateValue != null && dueDateValue != 0) {
//		var attrDueDate = values.createChild("ATTRIBUTE-VALUE-DATE");
//		attrDueDate.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-DATE-REF").createText("AD-DueDate-" + refSuffix);
//		attrDueDate.createAttribute("THE-VALUE", getISODateTimeWithSeconds(dueDateValue));
//	}

	// Custom Properties
	writeCustomProperties(values, IDENTIFIER.object.issue, issue);
	
	writeReqIFUserAttributeValue(values, refSuffix, {"cockpitObject" : issue});
	
	return identifier;
}

/**
 * Write a SpecObject for a comment
 * 
 * @param xmlNode
 * @param issue
 * 
 * @return the ReqIF Identifier of the written object
 */
function writeSpecObjectComment(xmlNode, issue) {
	var identifier = composeGUID(IDENTIFIER.object.comment + "-" + issue.getUniqueIdentifier());
	var date = getLastChangeValue(issue);
	
	var specObject = xmlNode.createChild("SPEC-OBJECT");
	
	specObject.createAttribute("IDENTIFIER", identifier);
	specObject.createAttribute("LONG-NAME", issue.getTitle());
	specObject.createAttribute("LAST-CHANGE", date);
	specObject.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-" + IDENTIFIER.object.comment);
	
	var values = specObject.createChild("VALUES");

	var attrDescription = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrDescription.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectText-" + IDENTIFIER.object.comment);
	
	var description = "";
	
	issue.getDescription().toArray().forEach(function(line) {
		description += line;
	});
	
	attrDescription.createAttribute("THE-VALUE", description);
	
	writeReqIFUserAttributeValue(values, "Comment", {"cockpitObject" : issue});
	
	return identifier;
}

/**
 * Write a relation
 * 
 * @param xmlNode
 * @param relationInfo Object with fields
 * <ul>
 * <li>identifier: identifier of the relation</li>
 * <li>longName: long name of the relation</li>
 * <li>lastChange: time of the last change of the relation</li>
 * <li>desc: description of the relation</li>
 * <li>relationSpecTypeRef: identfier of the spec type of the relation</li>
 * <li>sourceSpecObjectRef: identfier of the source spec object</li>
 * <li>targetSpecObjectRef: identfier of the target spec object</li>
 * </ul>
 */
function writeRelationIfObjectsExist(xmlNode, relationInfo) {
	if (relationInfo.targetSpecObjectRef && relationInfo.sourceSpecObjectRef &&
			typeof global.allExportedReqIFIdentifier[relationInfo.sourceSpecObjectRef] !== "undefined" &&
			typeof global.allExportedReqIFIdentifier[relationInfo.targetSpecObjectRef] !== "undefined") {
	
		var relation = xmlNode.createChild("SPEC-RELATION");
		relation.createAttribute("IDENTIFIER", composeGUID(relationInfo.identifier + "-" + global.relationCount));
		relation.createAttribute("LAST-CHANGE", relationInfo.lastChange);
		relation.createChild("SOURCE").createChild("SPEC-OBJECT-REF").createText(relationInfo.sourceSpecObjectRef);
		relation.createChild("TARGET").createChild("SPEC-OBJECT-REF").createText(relationInfo.targetSpecObjectRef);
		relation.createChild("TYPE").createChild("SPEC-RELATION-TYPE-REF").createText(relationInfo.relationSpecTypeRef);
	
		global.relationCount++;
	} else {
		LOGGER.info("Not writing relation from '" + String(relationInfo.sourceSpecObjectRef) + "' to not exported or missing object '" + String(relationInfo.targetSpecObjectRef) + "'");
	}
}

/**
 * Writes all requirement-to-requirement relations for a single requirement
 * 
 * @param xmlNode
 * @param req the requirment to write all requirement-to-requirement relations for
 */
function writeRelationRequirementToRequirement (xmlNode, req) {
	var relationInfo = new Object();
	
	// default values for all relations of this type
	relationInfo.identifier = IDENTIFIER.relation.RequirementReferencesRequirement;
	relationInfo.lastChange = templateTime;
	relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.RequirementReferencesRequirement;
	relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.requirement + "-" + req.getUniqueIdentifier());
	
	req.getReferredRequirements().toArray().forEach(function(target) {
		relationInfo.targetSpecObjectRef = composeGUID(IDENTIFIER.object.requirement + "-" + target.getUniqueIdentifier());
		writeRelationIfObjectsExist(xmlNode, relationInfo);
	});
}

/**
 * Writes all requirement-to-modelElement relations for a single requirement
 * 
 * @param xmlNode
 * @param req the requirment to write all requirement-to-modelElement relations for
 */
function writeRelationRequirementToModelElement(xmlNode, req) {
	var relationInfo = new Object();
	
	// default values for all relations of this type
	relationInfo.identifier = IDENTIFIER.relation.ModelElementLinksRequirement;
	relationInfo.lastChange = templateTime;
	relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.ModelElementLinksRequirement;
	relationInfo.targetSpecObjectRef = composeGUID(IDENTIFIER.object.requirement + "-" + req.getUniqueIdentifier());
	
	req.getLinkedModelElements().toArray().forEach(function(modelElement) {
		relationInfo.sourceSpecObjectRef = getRifIdentifierForModelElement(modelElement);
		writeRelationIfObjectsExist(xmlNode, relationInfo);
	});
}

/**
 * Writes all modelElement-contains-modelElement relations for a single model element
 * 
 * @param xmlNode
 * @param modelElement the model element to write all modelElement-contains-modelElement relations for
 */
function writeRelationModelElementContainsModelElement(xmlNode, modelElement) {
	var relationInfo = new Object();
	
	// default values for all relations of this type
	relationInfo.identifier = IDENTIFIER.relation.ModelElementContainsModelElement;
	relationInfo.lastChange = templateTime;
	relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.ModelElementContainsModelElement;
	relationInfo.sourceSpecObjectRef = getRifIdentifierForModelElement(modelElement);
	
	modelElement.getDirectlyContainedModelElements().toArray().forEach(function(containedElement) {
		relationInfo.targetSpecObjectRef = getRifIdentifierForModelElement(containedElement);	
		writeRelationIfObjectsExist(xmlNode, relationInfo);
	});
}

/**
 * Writes all plan-shows-modelElement relations for a single model plan
 * 
 * @param xmlNode
 * @param plan the plan to write all plan-shows-modelElement relations for
 */
function writeRelationPlanShowsModelElement(xmlNode, plan) {
	var relationInfo = new Object();
	
	// default values for all relations of this type
	relationInfo.identifier = IDENTIFIER.relation.PlanShowsModelElement;
	relationInfo.lastChange = templateTime;
	relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.PlanShowsModelElement;
	relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.plan + "-" + plan.getUniqueIdentifier());
	
	plan.getAllModelElements().toArray().forEach(function(shownElement){
		relationInfo.targetSpecObjectRef = getRifIdentifierForModelElement(shownElement);	
		writeRelationIfObjectsExist(xmlNode, relationInfo);
	});
}

/**
 * Writes all issue-concerns-modelElement relations for a single model issue
 * 
 * @param xmlNode
 * @param issue the issue to write all issue-concerns-modelElement relations for
 */
function writeRelationIssueConcernsModelElement(xmlNode, issue) {
	var relationInfo = new Object();
	
	// default values for all relations of this type
	relationInfo.lastChange = templateTime;
	
	if (String(issue.getCategoryID()) === OPTION.IssueCategoryForComments) {
		relationInfo.identifier = IDENTIFIER.relation.CommentAnnotatesObject;
		relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.CommentAnnotatesObject;
		relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.comment + "-" + issue.getUniqueIdentifier());
	} else {
		relationInfo.identifier = IDENTIFIER.relation.IssueLinksModelElement;
		relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.IssueLinksModelElement;
		relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.issue + "-" + issue.getUniqueIdentifier());
	}
	
	issue.getLinkedModelElements().toArray().forEach(function(target){
		relationInfo.targetSpecObjectRef = getRifIdentifierForModelElement(target);	
		writeRelationIfObjectsExist(xmlNode, relationInfo);
	});
}

/**
 * Writes all issue-concerns-requirement relations for a single issue
 * 
 * @param xmlNode
 * @param issue the issue to write all issue-concerns-requirement relations for
 */
function writeRelationIssueConcernsRequirement(xmlNode, issue) {
	var relationInfo = new Object();
	
	// default values for all relations of this type
	relationInfo.lastChange = templateTime;
	
	if (String(issue.getCategoryID()) === OPTION.IssueCategoryForComments) {
		relationInfo.identifier = IDENTIFIER.relation.CommentAnnotatesObject;
		relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.CommentAnnotatesObject;
		relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.comment + "-" + issue.getUniqueIdentifier());
	} else {
		relationInfo.identifier = IDENTIFIER.relation.IssueLinksRequirement;
		relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.IssueLinksRequirement;
		relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.issue + "-" + issue.getUniqueIdentifier());
	}
	
	cockpit.issueLinkReportProvider.getConcernedItems(issue, "gm.requirementsmodule3.requirement").toArray().forEach(function(target) {
		relationInfo.targetSpecObjectRef = composeGUID(IDENTIFIER.object.requirement + "-" + target.getUniqueIdentifier());
		writeRelationIfObjectsExist(xmlNode, relationInfo);
	});
}

/**
 * Writes all issue-concerns-issue relations for a single issue
 * 
 * @param xmlNode
 * @param issue the issue to write all issue-concerns-issues relations for
 */
function writeRelationIssueConcernsIssue(xmlNode, issue) {
	var relationInfo = new Object();
	
	// default values for all relations of this type
	relationInfo.lastChange = templateTime;
	
	if (String(issue.getCategoryID()) === OPTION.IssueCategoryForComments) {
		relationInfo.identifier = IDENTIFIER.relation.CommentAnnotatesObject;
		relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.CommentAnnotatesObject;
		relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.comment + "-" + issue.getUniqueIdentifier());
	} else {
		relationInfo.identifier = IDENTIFIER.relation.IssueLinksIssue;
		relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.IssueLinksIssue;
		relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.issue + "-" + issue.getUniqueIdentifier());
	}
	
	cockpit.issueLinkReportProvider.getConcernedItems(issue, "gm.issuemodule2.issue").toArray().forEach(function(target) {
		relationInfo.targetSpecObjectRef = composeGUID(IDENTIFIER.object.issue + "-" + target.getUniqueIdentifier());
		writeRelationIfObjectsExist(xmlNode, relationInfo);
	});
}

/**
 * Writes a relation as stated in the ReqIF Identifier custom property for a single object
 * 
 * @param xmlNode
 * @param object
 */
function writeRelationNonCockpitLink(xmlNode, object) {
	if (object.hasPropertyValue(OPTION.CustomPropertyReqIFIdentifier)) {
		var reqIFIdentifier = object.getPropertyAsString(OPTION.CustomPropertyReqIFIdentifier);
		var relationInfo = new Object();

		relationInfo.lastChange = templateTime;
		var cockpitObjectTypeID = String(object.getTypeID());
		if (cockpitObjectTypeID === "gm.issuemodule2.issue") {
			if (String(object.getCategoryID()) === OPTION.IssueCategoryForComments) {
				relationInfo.identifier = "RT-" + IDENTIFIER.relation.CommentAnnotatesObject;
				relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.CommentAnnotatesObject;
				relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.comment + "-" + object.getUniqueIdentifier());
			} else {
				relationInfo.identifier = "RT-" + IDENTIFIER.relation.NonCockpitRelation;
				relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.NonCockpitRelation;
				relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.issue + "-" + object.getUniqueIdentifier());
			}
		} else if (cockpitObjectTypeID === "gm.requirementsmodule3.requirement") {
			relationInfo.identifier = "RT-" + IDENTIFIER.relation.NonCockpitRelation;
			relationInfo.relationSpecTypeRef = "RT-" + IDENTIFIER.relation.NonCockpitRelation;
			relationInfo.sourceSpecObjectRef = composeGUID(IDENTIFIER.object.requirement + "-" + object.getUniqueIdentifier());
		} else {
			LOGGER.warn("Relations via ReqIFIdentifier not supported for cockpit type '" + cockpitObjectTypeID + "'");
			return;
		}

		relationInfo.targetSpecObjectRef = reqIFIdentifier;
		writeRelationIfObjectsExist(xmlNode, relationInfo);
	}
}

/**
 * Writes a SpecObject for a plan 
 * 
 * @param xmlNode
 * @param plan
 * 
 * @return the ReqIF Identifier of the written object
 */
function writeSpecObjectPlan(xmlNode, plan) {
	var identifier = composeGUID(IDENTIFIER.object.plan + "-" + plan.getUniqueIdentifier());
	var date = getLastChangeValue(plan);
	
	var categoryIDRif;
	if (plan.hasDefaultCategory()) {
		categoryIDRif = "";
	} else {
		categoryIDRif = "-" + plan.getCategoryID();
	}
	
	var refSuffix = IDENTIFIER.object.plan + categoryIDRif;
	
	var specObject = xmlNode.createChild("SPEC-OBJECT");
	specObject.createAttribute("IDENTIFIER", identifier);
	specObject.createAttribute("LONG-NAME", plan.getName());
	specObject.createAttribute("LAST-CHANGE", date);
	specObject.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-" + refSuffix);
	
	var values = specObject.createChild("VALUES");
	
	var attrHeading = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrHeading.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-" + refSuffix);
	attrHeading.createAttribute("THE-VALUE", plan.getName());
	
	var attrDescription = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrDescription.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectText-" + refSuffix);
	
	var description = "";
	plan.getDescription().forEach(function(line) {
		description += line;
	});
	attrDescription.createAttribute("THE-VALUE", description);
	
	var attrResponsible = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrResponsible.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-PlanType-" + refSuffix);
	
	var type = String(plan.getPlanType());
	var planType = "";
	switch(type) {
		case "fmc.bd":
			planType = LABEL.PlanTypeFMCBD;
			break;
		case "fmc.pn":
			planType = LABEL.PlanTypeFMCPN;
			break;
		case "fmc.er":
			planType = LABEL.PlanTypeFMCER;
			break;
		case "bpre.epc":
			planType = LABEL.PlanTypeBPREEPC;
			break;
		case "bpre.oc":
			planType = LABEL.PlanTypeBPREOC;
			break;
		case "bpmn.bpd":
			planType = LABEL.PlanTypeBPMNBPD;
			break;
		case "uml.cd":
			planType = LABEL.PlanTypeUMLCD;
			break;
		case "uml.sd":
			planType = LABEL.PlanTypeUMLSD;
			break;
		case "acm.pd":
			planType = LABEL.PlanTypeACMPD;
			break;
		case "gui":
			planType = LABEL.PlanTypeGUI;
			break;
		default:
			LOGGER.warn("Unknown plan type '" + type + "'");
			break;
	}
	
	attrResponsible.createAttribute("THE-VALUE", planType);

	// Export of the plan 
	var attrPlanFile = values.createChild("ATTRIBUTE-VALUE-XHTML");
	attrPlanFile.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-XHTML-REF").createText("AD-Diagram-" + refSuffix);
	var attrPlanFileDiv = attrPlanFile.createChild("THE-VALUE").createChild("xhtml:div");
	
	var file = getPath(plan.getPNGImage());
	if (file != null) {
		if (includeFileInZip(file.getName(), cockpit.tempImageDirectory, global.zip.folderInZipForFiles)) {
			var planFileObject = attrPlanFileDiv.createChild("xhtml:object");
			planFileObject.createAttribute("name", file.getName());
			planFileObject.createAttribute("type", "image/png");
			planFileObject.createAttribute("data", file.toString().replace(java.io.File.separator, "/"));
		} else {
			attrPlanFileDiv.createText("Plan image not included due to exceeded file size sum.");
		}
	} else {
		LOGGER.error("Could not create plan graphic for plan '" + plan.getName() + "'");
		attrPlanFileDiv.createText("File not found!");
	}
	
	// Custom Properties
	writeCustomProperties(values, IDENTIFIER.object.plan, plan);
	
	writeReqIFUserAttributeValue(values, refSuffix, {"cockpitObject" : plan});
	
	return identifier;
}

/**
 * Recursivly writes all SpecObjects for folders and plans that are in a folder
 * 
 * @param xmlNode
 * @param folder folder or null for root
 */
function writeSpecObjectsForFolderContents(xmlNode, folder) {
    var childFolder;
    var childPlans;
    
    if (folder == null) {
    	childFolder = global.project.project.getRootFolders().toArray();
    	childPlans = global.project.project.getRootPlans();
    } else {
    	childFolder = folder.getSubFolders().toArray();
    	childPlans = folder.getPlans();
    }
    
	childFolder.forEach(function(subFolder) {
	    var doExport = false;
	    if (subFolder.getName().startsWith("[")) {
	    	doExport = OPTION.ExportTemplateFolder;
	    } else {
	    	doExport = exportCockpitObject(subFolder);
	    }
		if (doExport) {
	    	var reqIfIdentifier = writeSpecObjectFolder(xmlNode, subFolder);
        	
	    	writeSpecObjectsForFolderContents(xmlNode, subFolder);

        	global.plan.exportedFolders[subFolder.getUniqueIdentifier()] = subFolder;
        	global.allExportedReqIFIdentifier[reqIfIdentifier] = true;
        }
	});
	
	childPlans.forEach(function(plan) {
        if (exportCockpitObject(plan)) {
        	var reqIfIdentifier = writeSpecObjectPlan(xmlNode, plan);
        	
        	global.plan.exportedPlans[plan.getUniqueIdentifier()] = plan;
        	global.allExportedReqIFIdentifier[reqIfIdentifier] = true;
        }
	});
}

/**
 * Recursivly writes all SpecObjects for issue sets and issues that are in a issue set
 * 
 * @param xmlNode
 * @param issueSet issue set or null for root
 */
function writeSpecObjectsForIssueSetContents(xmlNode, issueSet) {
	var childIssueSets;
    var childIssues;
    
    if (issueSet == null) {
    	childIssueSets = cockpit.issueDataProvider.getAllRootIssueSets().toArray();
    	childIssues = cockpit.issueDataProvider.getAllRootIssues().toArray();
    } else {
    	childIssueSets = issueSet.getIssueSetChildren().toArray();
    	childIssues = issueSet.getIssueChildren().toArray();
    }
    
    childIssueSets.forEach(function(childIssueSet) {

    	// The comments issue set itself should never be exported but the contents might
    	if (String(childIssueSet.getName()) === LABEL.CockpitIssueSetNameForReqIFComments) {
    		writeSpecObjectsForIssueSetContents(xmlNode, childIssueSet);	
    	} else if (exportCockpitObject(childIssueSet)) {
    		var reqIfIdentifier = writeSpecObjectIssueSet(xmlNode, childIssueSet);
    		
    		writeSpecObjectsForIssueSetContents(xmlNode, childIssueSet);
    		
    		global.issue.exportedIssueSets[childIssueSet.getUniqueIdentifier()] = childIssueSet;
    		global.allExportedReqIFIdentifier[reqIfIdentifier] = true;
    	}
	});
	
    childIssues.forEach(function(issue) {
        if (exportCockpitObject(issue)) {
        	var reqIfIdentifier;
        	if (String(issue.getCategoryID()) === OPTION.IssueCategoryForComments) {
        		reqIfIdentifier = writeSpecObjectComment(xmlNode, issue);
    		} else {
    			reqIfIdentifier = writeSpecObjectIssue(xmlNode, issue);
    		}
        	
        	global.issue.exportedIssues[issue.getUniqueIdentifier()] = issue;
        	
        	global.allExportedReqIFIdentifier[reqIfIdentifier] = true;
        }
	});
}

/**
 * Writes a SpecObject for a folder
 * 
 * @param xmlNode
 * @param folder
 * 
 * @return the ReqIF Identifier of the written object
 */
function writeSpecObjectFolder(xmlNode, folder) {
	var date = getLastChangeValue(folder);
	
	var categoryIDRif;
	if (folder.hasDefaultCategory()) {
		categoryIDRif = "";
	} else {
		categoryIDRif = "-" + folder.getCategoryID();
	}
	
	var refSuffix = IDENTIFIER.object.plan + "Folder" + categoryIDRif;
	
	var identifier = composeGUID(IDENTIFIER.object.plan + "Folder-" + folder.getUniqueIdentifier());
	
	var specObject = xmlNode.createChild("SPEC-OBJECT");
	specObject.createAttribute("IDENTIFIER", identifier);
	specObject.createAttribute("LONG-NAME", folder.getName());
	specObject.createAttribute("LAST-CHANGE", date);
	specObject.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-" + refSuffix);

	var values = specObject.createChild("VALUES");
	
	var attrHeading = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrHeading.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-" + refSuffix);
	attrHeading.createAttribute("THE-VALUE", folder.getName());

	var attrDescription = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrDescription.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectText-" + refSuffix);
	
	var description = "";
	
	folder.getDescription().forEach(function(line) {
		description += line;
	});
	
	attrDescription.createAttribute("THE-VALUE", description);
	
	writeReqIFUserAttributeValue(values, refSuffix, {"cockpitObject" : folder});
	
	// Create an ATTRIBUTE-VALUE per Custom Property
	writeCustomProperties(values, IDENTIFIER.object.plan + "Folder", folder);
	
	return identifier;
}

/**
 * Writes a SpecObject for a requirement set
 * 
 * @param xmlNode
 * @param reqSet
 * 
 * @return the ReqIF Identifier of the written object
 */
function writeSpecObjectRequirementSet(xmlNode, reqSet) {
	var date = getLastChangeValue(reqSet);
	
	var categoryIDRif;
	if (reqSet.hasDefaultCategory()) {
		categoryIDRif = "";
	} else {
		categoryIDRif = "-" + reqSet.getCategoryID();
	}
	
	var refSuffix = IDENTIFIER.object.requirement + "Folder" + categoryIDRif;
	
	var identifier = composeGUID(IDENTIFIER.object.requirement + "Folder-" + reqSet.getUniqueIdentifier());	
	
	var specObject = xmlNode.createChild("SPEC-OBJECT");

	specObject.createAttribute("IDENTIFIER", identifier);
	specObject.createAttribute("LONG-NAME", reqSet.getName());
	specObject.createAttribute("LAST-CHANGE", date);
	specObject.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-" + refSuffix);

	var values = specObject.createChild("VALUES");

	var attrHeading = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrHeading.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-" + refSuffix);
	attrHeading.createAttribute("THE-VALUE", reqSet.getName());

	var attrDescription = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrDescription.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectText-" + refSuffix);
	
	var description = "";
	reqSet.getDescription().toArray().forEach(function(line) {
		description += line;
	});
	attrDescription.createAttribute("THE-VALUE", description);
	
	writeReqIFUserAttributeValue(values, refSuffix, {"cockpitObject" : reqSet});

	// Create an ATTRIBUTE-VALUE per Custom Property
	writeCustomProperties(values, IDENTIFIER.object.requirement + "Folder", reqSet);
	
	return identifier;
}

/**
 * Recursivly writes all SpecObjects for requirement sets and requirements that are in a requirement set
 * 
 * @param xmlNode
 * @param requirementSet requirement set or null for root
 */
function writeSpecObjectsForRequirementSetContents(xmlNode, requirementSet) {
	var childRequirementSets;
    var childRequirements;
    
    if (requirementSet == null) {
    	childRequirementSets = cockpit.requirementDataProvider.getAllRootRequirementSets().toArray();
    	childRequirements = cockpit.requirementDataProvider.getAllRootRequirements().toArray();
    } else {
    	childRequirementSets = requirementSet.getRequirementSetChildren().toArray();
    	childRequirements = requirementSet.getRequirementChildren().toArray();
    }
    
    childRequirementSets.forEach(function(childRequirementSet) {
    	var doExport = false;
    	// differentiate between "normal" requirement sets and the sepcial requirement set for new requirements
    	if (String(childRequirementSet.getName()) === LABEL.CockpitRequirementsSetNameForReqIFNewRequirements) {
        	// export the requirement set for the new requirements only, if they are public by default
    		doExport = OPTION.NewRequirementsArePublic;
    	} else {
    		doExport = exportCockpitObject(childRequirementSet);
    	}
    	if (doExport) {
	    	var reqIfIdentifier = writeSpecObjectRequirementSet(xmlNode, childRequirementSet);
	    	
	    	writeSpecObjectsForRequirementSetContents(xmlNode, childRequirementSet);
	    	
	    	global.requirement.exportedRequirementSets[childRequirementSet.getUniqueIdentifier()] = childRequirementSet;
	    	global.allExportedReqIFIdentifier[reqIfIdentifier] = true;
	    }
	});
	
    childRequirements.forEach(function(requirement) {
        if (exportCockpitObject(requirement)) {
        	var reqIfIdentifier = writeSpecObjectRequirement(xmlNode, requirement);
        	
        	global.requirement.exportedRequirements[requirement.getUniqueIdentifier()] = requirement;
        	
        	global.allExportedReqIFIdentifier[reqIfIdentifier] = true;
        }
	});
}

/**
 * Writes a SpecObject for an issue set
 * 
 * @param xmlNode
 * @param issueSet
 * 
 * @return the ReqIF Identifier of the written object
 */
function writeSpecObjectIssueSet(xmlNode, issueSet) {
	var date = getLastChangeValue(issueSet);
	
	var categoryIDRif;
	if (issueSet.hasDefaultCategory()) {
		categoryIDRif = "";
	} else {
		categoryIDRif = "-" + issueSet.getCategoryID();
	}
	
	var refSuffix = IDENTIFIER.object.issue + "Folder" + categoryIDRif;
	
	var identifier = composeGUID(IDENTIFIER.object.issue + "Folder-" + issueSet.getUniqueIdentifier());
	
	var specObject = xmlNode.createChild("SPEC-OBJECT");

	specObject.createAttribute("IDENTIFIER", identifier);
	specObject.createAttribute("LONG-NAME", issueSet.getName());
	specObject.createAttribute("LAST-CHANGE", date);
	specObject.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-" + refSuffix);

	var values = specObject.createChild("VALUES");
	
	var attrHeading = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrHeading.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-" + refSuffix);
	attrHeading.createAttribute("THE-VALUE", issueSet.getName());

	var attrDescription = values.createChild("ATTRIBUTE-VALUE-STRING");
	attrDescription.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectText-" + refSuffix);
	
	var description = "";
	issueSet.getDescription().toArray().forEach(function(line) {
		description += line;
	});
	attrDescription.createAttribute("THE-VALUE", description);
	
	writeReqIFUserAttributeValue(values, refSuffix, {"cockpitObject" : issueSet});
	
	// Create an ATTRIBUTE-VALUE per Custom Property
	writeCustomProperties(values, IDENTIFIER.object.issue + "Folder", issueSet);
	
	return identifier;
}

/**
 * Writes the hierarchy for folders (sections), contents
 * 
 * @param xmlNode
 * @param folder
 * @param hierarchyLvl
 * @param counter
 */
function writeSpecHierarchyFolderContents(xmlNode, folder, hierarchyLvl, counter) {
	var childFolder;
    var childPlans;
    
    if (folder == null) {
    	childFolder = global.project.project.getRootFolders().toArray();
    	childPlans = global.project.project.getRootPlans();
    } else {
    	childFolder = folder.getSubFolders().toArray();
    	childPlans = folder.getPlans();
    }
    
    childFolder.forEach(function(subFolder) {
	    if (global.plan.exportedFolders[subFolder.getUniqueIdentifier()] !== undefined) {
	    	var hierachy = xmlNode.createChild("SPEC-HIERARCHY");
	    	hierachy.createAttribute("IDENTIFIER", composeGUID("Hierarchy-" + IDENTIFIER.object.plan + "Folder-" + subFolder.getUniqueIdentifier()));
	    	hierachy.createAttribute("LONG-NAME", subFolder.getName());
	    	hierachy.createAttribute("LAST-CHANGE", templateTime);
	    	hierachy.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(
	    			composeGUID(IDENTIFIER.object.plan + "Folder-" + subFolder.getUniqueIdentifier()));

	    	var children = hierachy.createChild("CHILDREN");
	    	
	    	writeSpecHierarchyFolderContents(children, subFolder, hierarchyLvl + 1, counter);
        }
	});
    
    childPlans.forEach(function(plan) {
        if (global.plan.exportedPlans[plan.getUniqueIdentifier()] !== undefined) {
        	var child = xmlNode.createChild("SPEC-HIERARCHY");
    		child.createAttribute("IDENTIFIER", composeGUID("Hierarchy-" + IDENTIFIER.object.plan + "-" + plan.getUniqueIdentifier()));
    		child.createAttribute("LONG-NAME", plan.getName());
    		child.createAttribute("LAST-CHANGE", templateTime);
    		child.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(
    				composeGUID(IDENTIFIER.object.plan + "-" + plan.getUniqueIdentifier()));
        }
	});
}

/**
 * Writes the hierarchy for requirements sets and requirements
 * 
 * @param xmlNode
 * @param requirementSet requirement set or null for root
 * @param hierarchyLvl
 * @param counter
 */
function writeSpecHierarchyRequirementSetContents(xmlNode, requirementSet, hierarchyLvl, counter) {
	var childRequirementSets;
    var childRequirements;
    
    if (requirementSet == null) {
    	childRequirementSets = cockpit.requirementDataProvider.getAllRootRequirementSets().toArray();
    	childRequirements = cockpit.requirementDataProvider.getAllRootRequirements().toArray();
    } else {
    	childRequirementSets = requirementSet.getRequirementSetChildren().toArray();
    	childRequirements = requirementSet.getRequirementChildren().toArray();
    }
    
    childRequirementSets.forEach(function(requirementSet) {
	    if (global.requirement.exportedRequirementSets[requirementSet.getUniqueIdentifier()] !== undefined) {
	    	var hierachy = xmlNode.createChild("SPEC-HIERARCHY");
	    	hierachy.createAttribute("IDENTIFIER", composeGUID("Hierarchy-" + IDENTIFIER.object.requirement + "Folder-" + requirementSet.getUniqueIdentifier()));
	    	hierachy.createAttribute("LONG-NAME", requirementSet.getName());
	    	hierachy.createAttribute("LAST-CHANGE", templateTime);
	    	hierachy.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(
	    			composeGUID(IDENTIFIER.object.requirement + "Folder-" + requirementSet.getUniqueIdentifier()));

	    	var children = hierachy.createChild("CHILDREN");
	    	
	    	writeSpecHierarchyRequirementSetContents(children, requirementSet, hierarchyLvl + 1, counter);
        }
	});
    
    childRequirements.forEach(function(requirement) {
        if (global.requirement.exportedRequirements[requirement.getUniqueIdentifier()] !== undefined) {
        	var child = xmlNode.createChild("SPEC-HIERARCHY");
        	child.createAttribute("IDENTIFIER", composeGUID("Hierarchy-" + IDENTIFIER.object.requirement + "-" + requirement.getUniqueIdentifier()));
        	child.createAttribute("LONG-NAME", requirement.getName());
        	child.createAttribute("LAST-CHANGE", templateTime);
        	child.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(
        			composeGUID(IDENTIFIER.object.requirement + "-" + requirement.getUniqueIdentifier()));
        }
	});
}

/**
 * Writes the hierarchy for issues sets and issues
 * 
 * @param xmlNode
 * @param issueSet issue set or null for root
 * @param hierarchyLvl
 * @param counter
 */
function writeSpecHierarchyIssueSetContents(xmlNode, issueSet, hierarchyLvl, counter) {
	var childIssueSets;
    var childIssues;
    
    if (issueSet == null) {
    	childIssueSets = cockpit.issueDataProvider.getAllRootIssueSets().toArray();
    	childIssues = cockpit.issueDataProvider.getAllRootIssues().toArray();
    } else {
    	childIssueSets = issueSet.getIssueSetChildren().toArray();
    	childIssues = issueSet.getIssueChildren().toArray();
    }
    
    childIssueSets.forEach(function(issueSet) {
	    if (global.issue.exportedIssueSets[issueSet.getUniqueIdentifier()] !== undefined) {
	    	var hierachy = xmlNode.createChild("SPEC-HIERARCHY");
	    	hierachy.createAttribute("IDENTIFIER", composeGUID("Hierarchy-" + IDENTIFIER.object.issue + "Folder-" + issueSet.getUniqueIdentifier()));
	    	hierachy.createAttribute("LONG-NAME", issueSet.getName());
	    	hierachy.createAttribute("LAST-CHANGE", templateTime);
	    	hierachy.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(
	    			composeGUID(IDENTIFIER.object.issue + "Folder-" + issueSet.getUniqueIdentifier()));

	    	var children = hierachy.createChild("CHILDREN");
	    	
	    	writeSpecHierarchyIssueSetContents(children, issueSet, hierarchyLvl + 1, counter);
        }
	});
    
    childIssues.forEach(function(issue) {
        if (global.issue.exportedIssues[issue.getUniqueIdentifier()] !== undefined) {
        	var child = xmlNode.createChild("SPEC-HIERARCHY");
        	child.createAttribute("IDENTIFIER", composeGUID("Hierarchy-" + IDENTIFIER.object.issue + "-" + issue.getUniqueIdentifier()));
        	child.createAttribute("LONG-NAME", issue.getTitle());
        	child.createAttribute("LAST-CHANGE", templateTime);
        	if (String(issue.getCategoryID()) === OPTION.IssueCategoryForComments) {
        		child.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(
        				composeGUID(IDENTIFIER.object.comment + "-" + issue.getUniqueIdentifier()));
        	} else {
        		child.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(
        				composeGUID(IDENTIFIER.object.issue + "-" + issue.getUniqueIdentifier()));
        	}
        }
	});
}

/**
 * Writes the stakeholders in a table
 * 
 * @param xmlNode
 * @param stakeholders
 */
function writeStakeholders(xmlNode, stakeholders) {
	xmlNode.createText(LABEL.StakeholderChapterTitle + ":");
	xmlNode.createChild("xhtml:br");
	var table = xmlNode.createChild("xhtml:table");
	table.createAttribute("border", "1");
	table.createAttribute("cellspacing", "0");
	table.createAttribute("cellpadding", "4");
	
	var row = table.createChild("xhtml:tr");
	row.createChild("xhtml:td").createText(LABEL.StakeholderName);
	row.createChild("xhtml:td").createText(LABEL.StakeholderComapany);
	row.createChild("xhtml:td").createText(LABEL.StakeholderPhone);
	row.createChild("xhtml:td").createText(LABEL.StakeholderEmail);

	// Iterate through all stakeholders of the project and create a list of all stakeholders
	stakeholders.forEach(function(stakeholder) {
		row = table.createChild("xhtml:tr");
		row.createChild("xhtml:td").createText(stakeholder.getDisplayName());
		row.createChild("xhtml:td").createText(stakeholder.getCompany());
		row.createChild("xhtml:td").createText(stakeholder.getPhone());
		row.createChild("xhtml:td").createText(stakeholder.getEmail());
	});  
}

/**
 * Creates a Java File Object from an xml string generated from the report interface
 * 
 * @param stringToExtractPathFrom the xml string that contains the path
 * 
 * @returns {java.io.File} the Java File Object or null if no path was found in the string
 */
function getPath(stringToExtractPathFrom) {
	if (stringToExtractPathFrom) {
		var startString = "url=\"";
		var start = stringToExtractPathFrom.indexOf("url=\"");
	
		if (start > -1) {
			var end = stringToExtractPathFrom.indexOf("\"", start + startString.length);
			if (end > start) {
				var extractedPath = stringToExtractPathFrom.substring(start + startString.length, end);
				var file = new java.io.File(extractedPath);
				return file;
			}
		}
	}
	return null;
}

/**
 * Marks a file to be included in the zip
 * 
 * @param fileName the name of the file including extension but excluding the directory
 * @param pathOnDisk the path of the file on the disk
 * @param pathInZip the (relative) path of the file in the zip
 * 
 * @return true, if the file is marked as to be included; false if it exceeds the maximum file size sum
 */
function includeFileInZip(fileName, pathOnDisk, pathInZip) {
	var fileObject = new java.io.File(pathOnDisk, fileName);
	if (fileObject.canRead()) {
		var fileSize = fileObject.length();
		if (global.zip.filesToIncludeSize + fileSize < OPTION.MaxFileSizeSumInMegaByte * 1048576) {
			global.zip.filesToIncludeSize += fileSize;
			var fileToZip = new Object();
			fileToZip.fileName = fileName;
			fileToZip.pathOnDisk = pathOnDisk;
			fileToZip.pathInZip = pathInZip;
			global.zip.filesToZip.push(fileToZip);
			return true;
		} else {
			LOGGER.warn("File '" + fileName + "' would exceed maximum file size sum of " + OPTION.MaxFileSizeSumInMegaByte + "MB" +
					" (" + global.zip.filesToIncludeSize / 1048576 + "MB exported so far). Not adding it to export!");
			return false;
		}
	} else {
		LOGGER.error("Cannot read file '" + fileObject.getAbsolutePath() + "'");
	}
	return false;
}

/**
 * Returns the time in ISO8601 format with seconds and time zone.<p>
 * 
 * This method is needed as Cockpit does not include seconds in its ISO8601 format
 * 
 * @param javaLong Java Long object
 * 
 * @returns string representing the time in ISO8601 format with seconds and time zone
 */
function getISODateTimeWithSeconds(javaLong) {
	if (javaLong != null) {
		var timeString = cockpit.dateTimeProvider.printTimestampInISO8601Format(javaLong, true);
		return timeString.substring(0, 16) + ":00" + timeString.substring(16);
	} else {
		return "1970-01-01T00:00:00+00:00";
	}
}

function getRifIdentifierForModelElement(modelElement) {
	var elementTypeID = modelElement.getElementTypeID().toString();
	var elementType;

	if (elementTypeID == "bpre.information") {
		elementType = IDENTIFIER.object.modelElementInformation;
	} else if (elementTypeID == "bpre.state") {
		elementType = IDENTIFIER.object.modelElementState;
	} else if (elementTypeID == "bpre.function") {
		elementType = IDENTIFIER.object.modelElementFunction;
	} else {
		throw new Error("Unknown model element type id '" + elementTypeID + "'");
	}
	
	var identifier = composeGUID(
			IDENTIFIER.object.modelElement + "-" +
			elementType + "-" +
			modelElement.getUniqueIdentifier()
	);
	
	return identifier;
}

/**
 * Checks if a cockpit object should be exported according to it's visibility status
 * 
 * @param cockpitObject
 * @return true or false
 */
function exportCockpitObject(cockpitObject) {
	if (cockpitObject.hasPropertyValue(OPTION.CustomPropertyExportVisibility)) {
		var javaBoolean = cockpitObject.getPropertyAsJavaObject(OPTION.CustomPropertyExportVisibility);
		return Boolean(javaBoolean.booleanValue());
	} else if (typeof OPTION.CustomPropertySyncVisibilityIsOptional === "boolean") {
		return OPTION.CustomPropertySyncVisibilityIsOptional;
	}
	return false;
}

/**
 * Checks if a custom property should be exported
 * 
 * @param propertyID
 * @return true or false
 */
function exportCustomProperty(propertyID) {
	var propertyIDString = String(propertyID);
	if (propertyIDString === OPTION.CustomPropertyExportVisibility ||
			propertyIDString === OPTION.CustomPropertyReqIFIdentifier ||
			propertyIDString === OPTION.CustomPropertyReferredElement ||
			propertyIDString === OPTION.CustomPropertyReferredElementURL ||
			propertyIDString === OPTION.CustomPropertyCreatedBy ||
			propertyIDString === OPTION.CustomPropertyCreatedAt) {
		return false;
	}
	if (Array.isArray(OPTION.CustomPropertiesToExclude) && OPTION.CustomPropertiesToExclude.indexOf(propertyIDString) >= 0) {
		return false;
	}
	return true;
}

/**
 * Returns the date a cockpit object was changed the last time.
 * 
 * @param cockpitObject
 */
function getLastChangeValue(cockpitObject) {
	var dateInMilliSec = 0;
	if (cockpitObject.hasPropertyValue(OPTION.CustomPropertyCreatedAt)) {
		var dateString = cockpitObject.getPropertyAsString(OPTION.CustomPropertyCreatedAt);
		var javaDate = parseISODateFormat(String(dateString));
		if (javaDate != null) {
			dateInMilliSec = Number(javaDate.getTime());
		} else {
			LOGGER.warn("Cannot parse ISO 8601 date '" + dateString + "'.");
		}
	}
	if (!dateInMilliSec) {
		dateInMilliSec = Number(cockpitObject.getDateOfLastModification_MilliSecondsSince_01_01_1970());
	}
	if (dateInMilliSec < 86400000) {
		dateInMilliSec = 86400000; // adding 24h to ensure it's a valid date after 1970-01-01 no matter which timezone
	}
	return getISODateTimeWithSeconds(dateInMilliSec);
}

/**
 * Parses in ISO 8601 date in its longest form with date, time and time zone and returns a Java Date object
 * 
 * @param dateAsString
 * 
 * @return Java Date object or null
 */
function parseISODateFormat(dateAsString) {

	if (typeof dateAsString !== "string") {
		return null;
	}
	
	if (dateAsString.length != 25 && dateAsString.length != 20) {
		return null;
	}
	
	var dateFormat = new java.text.SimpleDateFormat( "yyyy-MM-dd'T'HH:mm:ssz" );

	//this is zero time so we need to add that TZ indicator for 
	if (dateAsString.charAt(dateAsString.length - 1) === "Z") {
		dateAsString = dateAsString.substring(0, dateAsString.length - 1) + "GMT-00:00";
	} else {
		var subStringDateTime = dateAsString.substring(0, dateAsString.length - 6);
		var subStringTimeZone = dateAsString.substring(dateAsString.length - 6, dateAsString.length);
		dateAsString = subStringDateTime + "GMT" + subStringTimeZone;
	}

	try {
		return dateFormat.parse(dateAsString);
	} catch (e) {
		return null;
	}
}

function writeReqIFUserAttributeType(xmlNode, attributeTypeSuffix) {
	var specAttribute = xmlNode.createChild("ATTRIBUTE-DEFINITION-STRING");
	specAttribute.createAttribute("IDENTIFIER", "AD-CreatedBy-" + attributeTypeSuffix);
	specAttribute.createAttribute("LONG-NAME", LABEL.CreatedByUsername);
	specAttribute.createAttribute("LAST-CHANGE", templateTime);
	specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
}

function writeReqIFUserAttributeValue(xmlNode, attributeTypeSuffix, usernameOrObject) {
	var username = "";
	if (typeof usernameOrObject === "object" && usernameOrObject != null) {
		if (usernameOrObject.hasOwnProperty("username")) {
			username = usernameOrObject.username;
		} else if (usernameOrObject.hasOwnProperty("cockpitObject")) {
			if (usernameOrObject.cockpitObject.hasPropertyValue(OPTION.CustomPropertyCreatedBy)) {
				username = usernameOrObject.cockpitObject.getPropertyAsString(OPTION.CustomPropertyCreatedBy);
			} else {
				username = usernameOrObject.cockpitObject.getCreator();
			}
		} else {
			assert(false, "Neither username nor cockpit object given.");
		}
	}
	
	var attrUsername = xmlNode.createChild("ATTRIBUTE-VALUE-STRING");
	attrUsername.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-CreatedBy-" + attributeTypeSuffix);
	attrUsername.createAttribute("THE-VALUE", username);
}

/**
 * Extends a project unique identifier with the project identifier to form a global unique identifier
 * 
 * @param projectGlobalIdentifier identifier that is unique within the project
 * 
 * @return global unique identifier
 */
function composeGUID(projectGlobalIdentifier) {
	return projectGlobalIdentifier + "-" + global.project.reqIfIdentifier;
}

/**
 * Returns the mime type for a file name by evaluating the file extension
 * 
 * @param filename file name including extension
 * 
 * @returns mime time (e.g. image/png) or null if none was found
 */
function mimeTypeForExtension(fileName) {
	var lowerFilename = fileName.toLowerCase();
	if (lowerFilename.endsWith(".png")) {
		return "image/png";
	} else if (lowerFilename.endsWith(".jpg") || lowerFilename.endsWith(".jpeg")) {
		return "image/jpeg";
	} else if (lowerFilename.endsWith(".ico")) {
		return "image/x-icon";
	} else if (lowerFilename.endsWith(".bmp")) {
		return "image/bmp";
	} else if (lowerFilename.endsWith(".gif")) {
		return "image/gif";
	}
	return null;
}

/**
 * Returns whether the file name is potentially an image by evaluating the file extension
 * 
 * @param fileName file name including extension
 * 
 * @returns true if the file name is an image
 */
function isImage(fileName) {
	var lowerFilename = fileName.toLowerCase();
	return (lowerFilename.endsWith(".png") || lowerFilename.endsWith(".jpg") || lowerFilename.endsWith(".jpeg") ||
			lowerFilename.endsWith(".ico") || lowerFilename.endsWith(".bmp") || lowerFilename.endsWith(".gif"));
}

/* ---------------- Script Vars ------------------ */
var templateTime = "2013-05-01T00:00:00+01:00";

/* ---------------- Global Vars ------------------ */
global.relationCount = 1;

global.zip = new Object();
global.zip.filesToZip = []; // List of Objects that have the fields "fileName", "pathInZip", and "pathOnDisk"
global.zip.filesToIncludeSize = 0;
global.zip.folderInZipForFiles = "files_and_images";

global.requirement = new Object();
global.requirement.allStatusValues = cockpit.requirementDataProvider.getKnownValuesForRequirementStatus();
global.requirement.allPrioValues = cockpit.requirementDataProvider.getKnownValuesForRequirementPriority();
global.requirement.exportedRequirements = new Object();
global.requirement.exportedRequirementSets = new Object();

global.issue = new Object();
global.issue.allStatusValues = cockpit.issueDataProvider.getKnownValuesForIssueStatus();
global.issue.allPrioValues = cockpit.issueDataProvider.getKnownValuesForIssuePriority();
global.issue.exportedIssues = new Object();
global.issue.exportedIssueSets = new Object();

global.plan = new Object();
global.plan.exportedPlans = new Object();
global.plan.exportedFolders = new Object();

global.modelElement = new Object();
global.modelElement.exportedModelElementsArray = [];

global.stakeholders = new Object();
global.stakeholders.allStakeholders = global.project.project.getAllStakeholders();

global.allExportedReqIFIdentifier = new Object();

/* ====================== Main ======================== */
var actualDateTimeObject = cockpit.dateTimeProvider.getCurrentDateTime_MilliSecondsSince_01_01_1970();
var actualDateTimeISO = getISODateTimeWithSeconds(actualDateTimeObject);

LOGGER.debug("Starting XML Document.");

/*
 * New XML Document.
 * The escaping for special characters is disabled, as the report interface already returns escaped content
 */ 
var doc = new XMLDocument("REQ-IF", true);
var rifroot = doc.rootNode;

rifroot.createAttribute("xmlns", "http://www.omg.org/spec/ReqIF/20110401/reqif.xsd");
rifroot.createAttribute("xmlns:xhtml", "http://www.w3.org/1999/xhtml");

/* ----------- Header ------------ */
LOGGER.debug("Writing REQ-IF Header.");
(function(){
	var header = rifroot.createChild("THE-HEADER").createChild("REQ-IF-HEADER");
	header.createAttribute("IDENTIFIER", global.project.reqIfIdentifier);
	header.createChild("COMMENT").createText("System Specification of Project '" + global.project.project.getName() + "'.\n" + 
			"It contains:\n" + 
			"* plans,\n" +
			"* model elements,\n" +
			"* requirements,\n" +
			"* open issues,\n" +
			"* occurrences of model elements on plans,\n" +
			"* relations between requirements and model elements,\n" +
			"* dependencies between requirements, as well as\n" +
			"* containment of model-elements");
	header.createChild("CREATION-TIME").createText(actualDateTimeISO);
	header.createChild("REQ-IF-TOOL-ID").createText("ARCWAY Cockpit with CoPilot");
	header.createChild("REQ-IF-VERSION").createText("1.0");
	header.createChild("SOURCE-TOOL-ID").createText("ARCWAY Cockpit with CoPilot");
	header.createChild("TITLE").createText(global.project.project.getDisplayName());	
	
})();

// ---------- Content----------
var rifContent = rifroot.createChild("CORE-CONTENT").createChild("REQ-IF-CONTENT");

// Data types
LOGGER.debug("Writing REQ-IF DataTypes.");
var rifDataTypes = rifContent.createChild("DATATYPES");

(function() {
	var rifDataType;
	// Type String
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-STRING");
	rifDataType.createAttribute("IDENTIFIER", "DT-String");
	rifDataType.createAttribute("LONG-NAME", "String[255] DataType");
	rifDataType.createAttribute("DESC", "This is a standard sting type");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	rifDataType.createAttribute("MAX-LENGTH", "255");
	
	// Type Text
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-STRING");
	rifDataType.createAttribute("IDENTIFIER", "DT-Text");
	rifDataType.createAttribute("LONG-NAME", "Text DataType");
	rifDataType.createAttribute("DESC", "This is a standard text type");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	rifDataType.createAttribute("MAX-LENGTH", "32000");
	
	// Type Formatted Text
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-XHTML");
	rifDataType.createAttribute("IDENTIFIER", "DT-FormattedText");
	rifDataType.createAttribute("LONG-NAME", "Formatted Text");
	rifDataType.createAttribute("DESC", "This is a formatted text type");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	
	// Type Boolean
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-BOOLEAN");
	rifDataType.createAttribute("IDENTIFIER", "DT-Boolean");
	rifDataType.createAttribute("LONG-NAME", "Boolean DataType");
	rifDataType.createAttribute("DESC", "This is a standard boolean type");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	
	// Type Integer
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-INTEGER");
	rifDataType.createAttribute("IDENTIFIER", "DT-Integer");
	rifDataType.createAttribute("LONG-NAME", "Integer DataType");
	rifDataType.createAttribute("DESC", "This is a standard signed integer type");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	rifDataType.createAttribute("MIN", String(java.lang.Integer.MIN_VALUE));
	rifDataType.createAttribute("MAX", String(java.lang.Integer.MAX_VALUE));
	
	// Type Real
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-REAL");
	rifDataType.createAttribute("IDENTIFIER", "DT-Real");
	rifDataType.createAttribute("LONG-NAME", "Real DataType");
	rifDataType.createAttribute("DESC", "This is a standard real type");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	rifDataType.createAttribute("MIN", "-" + String(java.lang.Double.MAX_VALUE));
	rifDataType.createAttribute("MAX", String(java.lang.Double.MAX_VALUE));
//	rifDataType.createAttribute("MIN", "-" + String(1000000000));
//	rifDataType.createAttribute("MAX", String(1000000000));
	rifDataType.createAttribute("ACCURACY", "15");
	
	// Type ISO Date
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-DATE");
	rifDataType.createAttribute("IDENTIFIER", "DT-ISODate");
	rifDataType.createAttribute("LONG-NAME", "ISO Date-Time DataType");
	rifDataType.createAttribute("DESC", "This is a standard date-time type in ISO Format");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	
	// Type URL
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-STRING");
	rifDataType.createAttribute("IDENTIFIER", "DT-URL");
	rifDataType.createAttribute("LONG-NAME", "URL");
	rifDataType.createAttribute("DESC", "This is a standard URL");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	rifDataType.createAttribute("MAX-LENGTH", "255");
	
	// Type Enumeration for Status of Requirement
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-ENUMERATION");
	rifDataType.createAttribute("IDENTIFIER", "DT-" + IDENTIFIER.object.requirement + "-status");
	rifDataType.createAttribute("LONG-NAME", "Status");
	rifDataType.createAttribute("DESC", "Enumerated Values for the requirement status");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	
	var valuesCount = 0;
	var	values = rifDataType.createChild("SPECIFIED-VALUES");
	global.requirement.allStatusValues.toArray().forEach(function(val) {
		writeEnumValue(values, val, "V-" + IDENTIFIER.object.requirement + "-status-" + valuesCount, templateTime, valuesCount);
		valuesCount++;
	});
	
	// Type Enumeration for Priorities of Requirement
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-ENUMERATION");
	rifDataType.createAttribute("IDENTIFIER", "DT-" + IDENTIFIER.object.requirement + "-priority");
	rifDataType.createAttribute("LONG-NAME", "Priority");
	rifDataType.createAttribute("DESC", "Enumerated Values for the requirement priority");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	
	valuesCount = 0;
	values = rifDataType.createChild("SPECIFIED-VALUES");
	global.requirement.allPrioValues.toArray().forEach(function(val) {
		writeEnumValue(values, val, "V-" + IDENTIFIER.object.requirement + "-prio-" + valuesCount, templateTime, valuesCount);
		valuesCount++;
	});
	
	// Type Enumeration for custom properties of issues
	writeEnumCustomPropertyDatatypeDefs(rifDataTypes, "gm.issuemodule2.issue", IDENTIFIER.object.issue);
	
	// Type Enumeration for custom properties of requirements
	writeEnumCustomPropertyDatatypeDefs(rifDataTypes, "gm.requirementsmodule3.requirement", IDENTIFIER.object.requirement);
	
	// Type Enumeration for custom properties of plans
	writeEnumCustomPropertyDatatypeDefs(rifDataTypes, "plan", IDENTIFIER.object.plan);
	
	// Type Enumeration for custom properties of model elements
	writeEnumCustomPropertyDatatypeDefs(rifDataTypes, "uniqueElement", IDENTIFIER.object.modelElement);
	
	// Type Enumeration for Status of Issues
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-ENUMERATION");
	rifDataType.createAttribute("IDENTIFIER", "DT-" + IDENTIFIER.object.issue + "-status");
	rifDataType.createAttribute("LONG-NAME", "Status");
	rifDataType.createAttribute("DESC", "Enumerated Values for the issue status");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	
	valuesCount = 0;
	values = rifDataType.createChild("SPECIFIED-VALUES");
	global.issue.allStatusValues.toArray().forEach(function(val) {
		writeEnumValue(values, val, "V-" + IDENTIFIER.object.issue + "-status-" + valuesCount, templateTime, valuesCount);
		valuesCount++;
	});
	
	// Type Enumeration for Priority of Issues
	rifDataType = rifDataTypes.createChild("DATATYPE-DEFINITION-ENUMERATION");
	rifDataType.createAttribute("IDENTIFIER", "DT-" + IDENTIFIER.object.issue + "-priority");
	rifDataType.createAttribute("LONG-NAME", "Priority");
	rifDataType.createAttribute("DESC", "Enumerated Values for the issue priority");
	rifDataType.createAttribute("LAST-CHANGE", templateTime);
	
	valuesCount = 0;
	values = rifDataType.createChild("SPECIFIED-VALUES");
	global.issue.allPrioValues.toArray().forEach(function(val) {
		writeEnumValue(values, val, "V-" + IDENTIFIER.object.issue + "-prio-" + valuesCount, templateTime, valuesCount);
		valuesCount++;
	});
})();

/* --------------- Spec Types --------------- */
LOGGER.debug("Writing REQ-IF SpecTypes.");
var rifSpecTypes = rifContent.createChild("SPEC-TYPES");

(function(){
	
	// Spec Types for plans and their categories
	var allCategories = cockpit.dataModelProvider.getCategoryIDsForObjectType("plan");
	allCategories.add(0, null); // adding default category
	
	allCategories.toArray().forEach(function(categoryID) {
		var categoryIDRif;
		var longNameAddition;
		if (categoryID == null) {
			categoryIDRif = "";
			longNameAddition = "with default category";
		} else {
			categoryIDRif = "-" + categoryID;
			longNameAddition = "with category '" + cockpit.dataModelProvider.getDisplaynameOfCategory("plan", categoryID) + "'";
		}
		
		var suffix = IDENTIFIER.object.plan + categoryIDRif;
		
		var specType = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
		specType.createAttribute("IDENTIFIER", "OT-" + suffix);
		specType.createAttribute("LONG-NAME", "Plan " + longNameAddition);
		specType.createAttribute("DESC", "A plan showing a usage scenario (business process) or the system topology.");
		specType.createAttribute("LAST-CHANGE", templateTime);
		
		var specAttributes = specType.createChild("SPEC-ATTRIBUTES");
		
		var specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.Name);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectText-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.Description);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-PlanType-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.PlanType);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
	
		// Attribute for referencing the plan diagram
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-XHTML");
		specAttribute.createAttribute("IDENTIFIER", "AD-Diagram-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.Diagram);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-XHTML-REF").createText("DT-FormattedText");
		
		writeCustomPropertyTypes(specAttributes, IDENTIFIER.object.plan, "plan", categoryID);
		
		writeReqIFUserAttributeType(specAttributes, suffix);
	});
	
	
	// Spec Types for requirements and their categories
	allCategories = cockpit.dataModelProvider.getCategoryIDsForObjectType("gm.requirementsmodule3.requirement");
	allCategories.add(0, null); // adding default category
	
	allCategories.toArray().forEach(function(categoryID) {
		var categoryIDRif;
		var longNameAddition;
		if (categoryID == null) {
			categoryIDRif = "";
			longNameAddition = "with default category";
		} else {
			categoryIDRif = "-" + categoryID;
			longNameAddition = "with category '" + cockpit.dataModelProvider.getDisplaynameOfCategory("gm.requirementsmodule3.requirement", categoryID) + "'";
		}
		
		var suffix = IDENTIFIER.object.requirement + categoryIDRif;
		
		var specType = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
		specType.createAttribute("IDENTIFIER", "OT-" + suffix);
		specType.createAttribute("LONG-NAME", "Requirement " + longNameAddition);
		specType.createAttribute("DESC", "A singular documented need of what a particular product or service should be or perform.");
		specType.createAttribute("LAST-CHANGE", templateTime);
		
		var specAttributes = specType.createChild("SPEC-ATTRIBUTES");
		
		var specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectID-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.ID);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
		
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.Name);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectText-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.Description);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-ENUMERATION");
		specAttribute.createAttribute("IDENTIFIER", "AD-Status-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.RequirementStatus);
		specAttribute.createAttribute("DESC", "Requirement Status");
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createAttribute("MULTI-VALUED", "false");
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-ENUMERATION-REF").createText("DT-" + IDENTIFIER.object.requirement + "-status");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-ENUMERATION");
		specAttribute.createAttribute("IDENTIFIER", "AD-Priority-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.RequirementPriority);
		specAttribute.createAttribute("DESC", "Requirement Priority");
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createAttribute("MULTI-VALUED", "false");
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-ENUMERATION-REF").createText("DT-" + IDENTIFIER.object.requirement + "-priority");
		
		writeCustomPropertyTypes(specAttributes, IDENTIFIER.object.requirement, "gm.requirementsmodule3.requirement", categoryID);
		
		writeReqIFUserAttributeType(specAttributes, suffix);
	});
	
	// Spec Types for unique elements and their categories
	allCategories = cockpit.dataModelProvider.getCategoryIDsForObjectType("uniqueElement");
	allCategories.add(0, null); // adding default category
	
	allCategories.toArray().forEach(function(categoryID) {
		var categoryIDRif;
		var longNameAddition;
		if (categoryID == null) {
			categoryIDRif = "";
			longNameAddition = "with default category";
		} else {
			categoryIDRif = "-" + categoryID;
			longNameAddition = "with category '" + cockpit.dataModelProvider.getDisplaynameOfCategory("uniqueElement", categoryID) + "'";
		}
		
		var suffix = IDENTIFIER.object.modelElement + categoryIDRif;
		
		var specType = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
		specType.createAttribute("IDENTIFIER", "OT-" + suffix);
		specType.createAttribute("LONG-NAME", "Model Element " + longNameAddition);
		specType.createAttribute("DESC", "A system function (also process step), information container or event.");
		specType.createAttribute("LAST-CHANGE", templateTime);
		
		var specAttributes = specType.createChild("SPEC-ATTRIBUTES");
		
		var specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.Name);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectText-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.Description);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-Type-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.ModelElementType);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
		
		writeCustomPropertyTypes(specAttributes, IDENTIFIER.object.modelElement, "uniqueElement", categoryID);
		
		writeReqIFUserAttributeType(specAttributes, IDENTIFIER.object.modelElement + categoryIDRif);
	});
	
	// Spec Types for issues and their categories
	allCategories = cockpit.dataModelProvider.getCategoryIDsForObjectType("gm.issuemodule2.issue");
	allCategories.add(0, null); // adding default category
	
	allCategories.toArray().forEach(function(categoryID) {
		// Comments will be handled separately
		if (String(categoryID) === OPTION.IssueCategoryForComments) {
			return;
		}
		var categoryIDRif;
		var longNameAddition;
		if (categoryID == null) {
			categoryIDRif = "";
			longNameAddition = "with default category";
		} else {
			categoryIDRif = "-" + categoryID;
			longNameAddition = "with category '" + cockpit.dataModelProvider.getDisplaynameOfCategory("gm.issuemodule2.issue", categoryID) + "'";
		}
		
		var suffix = IDENTIFIER.object.issue + categoryIDRif;
		
		var specType = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
		specType.createAttribute("IDENTIFIER", "OT-" + suffix);
		specType.createAttribute("LONG-NAME", "Open Issue " + longNameAddition);
		specType.createAttribute("DESC", "");
		specType.createAttribute("LAST-CHANGE", templateTime);
		
		var specAttributes = specType.createChild("SPEC-ATTRIBUTES");
		
		var specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectID-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.ID);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
		
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.Title);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-ObjectText-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.Description);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-ENUMERATION");
		specAttribute.createAttribute("IDENTIFIER", "AD-Status-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.IssueStatus);
		specAttribute.createAttribute("DESC", "Issue Status");
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createAttribute("MULTI-VALUED", "false");
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-ENUMERATION-REF").createText("DT-" + IDENTIFIER.object.issue + "-status");
	
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-ENUMERATION");
		specAttribute.createAttribute("IDENTIFIER", "AD-Priority-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.IssuePriority);
		specAttribute.createAttribute("DESC", "Issue Priority");
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createAttribute("MULTI-VALUED", "false");
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-ENUMERATION-REF").createText("DT-" + IDENTIFIER.object.issue + "-priority");
		
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
		specAttribute.createAttribute("IDENTIFIER", "AD-Responsible-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.IssueResponsible);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");

		// There currently is no way to have differently formatted datatype DATE dates so treat them as type STRING
		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
//		specAttribute = specAttributes.createChild("ATTRIBUTE-DEFINITION-DATE");
		specAttribute.createAttribute("IDENTIFIER", "AD-DueDate-" + suffix);
		specAttribute.createAttribute("LONG-NAME", LABEL.IssueDueDate);
		specAttribute.createAttribute("LAST-CHANGE", templateTime);
		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
//		specAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-DATE-REF").createText("DT-ISODate");
		
		writeCustomPropertyTypes(specAttributes, IDENTIFIER.object.issue, "gm.issuemodule2.issue", categoryID);
		
		writeReqIFUserAttributeType(specAttributes, IDENTIFIER.object.issue + categoryIDRif);
	});
})();

// SpecType for comments
(function() {
	var specTypeComment = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
	specTypeComment.createAttribute("IDENTIFIER", "OT-" + IDENTIFIER.object.comment);
	specTypeComment.createAttribute("LONG-NAME", "Comment");
	specTypeComment.createAttribute("DESC", "Comment that has been previously created via the web interface and was already transfered to the Cockpit Server.");
	specTypeComment.createAttribute("LAST-CHANGE", templateTime);
	
	var specTypeCommentAttrs = specTypeComment.createChild("SPEC-ATTRIBUTES");
	
	var specTypeCommentAttr = specTypeCommentAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
	specTypeCommentAttr.createAttribute("IDENTIFIER", "AD-ObjectText-" + IDENTIFIER.object.comment);
	specTypeCommentAttr.createAttribute("LONG-NAME", LABEL.Text);
	specTypeCommentAttr.createAttribute("DESC", "Text of the comment.");
	specTypeCommentAttr.createAttribute("LAST-CHANGE", templateTime);
	specTypeCommentAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
	writeReqIFUserAttributeType(specTypeCommentAttrs, "Comment");
})();

// SpecType for new comments
(function() {
	var specTypeComment = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
	specTypeComment.createAttribute("IDENTIFIER", "OT-" + IDENTIFIER.object.newComment);
	specTypeComment.createAttribute("LONG-NAME", "New Comment");
	specTypeComment.createAttribute("DESC", "New comment that was created via the web interface.");
	specTypeComment.createAttribute("LAST-CHANGE", templateTime);
	
	var specTypeCommentAttrs = specTypeComment.createChild("SPEC-ATTRIBUTES");
	
	var specTypeCommentAttr = specTypeCommentAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
	specTypeCommentAttr.createAttribute("IDENTIFIER", "AD-ObjectText-" + IDENTIFIER.object.newComment);
	specTypeCommentAttr.createAttribute("LONG-NAME", LABEL.Text);
	specTypeCommentAttr.createAttribute("DESC", "Text of the comment.");
	specTypeCommentAttr.createAttribute("LAST-CHANGE", templateTime);
	specTypeCommentAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
	writeReqIFUserAttributeType(specTypeCommentAttrs, "NewComment");
})();

// SpecType for New Requirements
(function() {
	var specType = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
	specType.createAttribute("IDENTIFIER", "OT-" + IDENTIFIER.object.newRequirement);
	specType.createAttribute("LONG-NAME", "New Requirement");
	specType.createAttribute("DESC", "Objects of this type represent requirement suggestions that have been entered by the ReqIF Server users.");
	specType.createAttribute("LAST-CHANGE", templateTime);
	
	var attributes = specType.createChild("SPEC-ATTRIBUTES");
	
	var title = attributes.createChild("ATTRIBUTE-DEFINITION-STRING");
	title.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + IDENTIFIER.object.newRequirement);
	title.createAttribute("LONG-NAME", LABEL.Name);
	title.createAttribute("DESC", "Name of the new requirement.");
	title.createAttribute("LAST-CHANGE", templateTime);
	title.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
	
	var text = attributes.createChild("ATTRIBUTE-DEFINITION-STRING");
	text.createAttribute("IDENTIFIER", "AD-ObjectText-" + IDENTIFIER.object.newRequirement);
	text.createAttribute("LONG-NAME", LABEL.Description);
	text.createAttribute("DESC", "Text of the new requirement.");
	text.createAttribute("LAST-CHANGE", templateTime);
	text.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
	writeReqIFUserAttributeType(attributes, "NewRequirement");
})();

// SpecType for New Issues
(function() {
	var specType = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
	specType.createAttribute("IDENTIFIER", "OT-" + IDENTIFIER.object.newIssue);
	specType.createAttribute("LONG-NAME", "New Issue");
	specType.createAttribute("DESC", "Objects of this type represent issues that have been entered by the ReqIF Server users.");
	specType.createAttribute("LAST-CHANGE", templateTime);
	
	var attributes = specType.createChild("SPEC-ATTRIBUTES");
	
	var title = attributes.createChild("ATTRIBUTE-DEFINITION-STRING");
	title.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + IDENTIFIER.object.newIssue);
	title.createAttribute("LONG-NAME", LABEL.Title);
	title.createAttribute("DESC", "Title of the new issue.");
	title.createAttribute("LAST-CHANGE", templateTime);
	title.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
	var text = attributes.createChild("ATTRIBUTE-DEFINITION-STRING");
	text.createAttribute("IDENTIFIER", "AD-ObjectText-" + IDENTIFIER.object.newIssue);
	text.createAttribute("LONG-NAME", LABEL.Description);
	text.createAttribute("DESC", "Text of the new issue.");
	text.createAttribute("LAST-CHANGE", templateTime);
	text.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
	writeReqIFUserAttributeType(attributes, "NewIssue");
})();

// SpecType for Settings
(function() {
	var suffix = "Setting";	
	
	var specType = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
	specType.createAttribute("IDENTIFIER", "OT-" + suffix);
	specType.createAttribute("LONG-NAME", "Setting");
	specType.createAttribute("DESC", "Type representing settings.");
	specType.createAttribute("LAST-CHANGE", templateTime);
	
	var attributes = specType.createChild("SPEC-ATTRIBUTES");
	
	var title = attributes.createChild("ATTRIBUTE-DEFINITION-STRING");
	title.createAttribute("IDENTIFIER", "AD-Value-" + suffix);
	title.createAttribute("LONG-NAME", "Value");
	title.createAttribute("DESC", "Value of the setting.");
	title.createAttribute("LAST-CHANGE", templateTime);
	title.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
})();

(function(){
	// Folders
	
	var allCategories = cockpit.dataModelProvider.getCategoryIDsForObjectType("folder");
	allCategories.add(0, null); // adding default category
	
	allCategories.toArray().forEach(function(categoryID) {
		var categoryIDRif;
		var longNameAddition;
		if (categoryID == null) {
			categoryIDRif = "";
			longNameAddition = "with default category";
		} else {
			categoryIDRif = "-" + categoryID;
			longNameAddition = "with category '" + cockpit.dataModelProvider.getDisplaynameOfCategory("folder", categoryID) + "'";
		}
		
		var suffix = IDENTIFIER.object.plan + "Folder" + categoryIDRif;
	
		var specTypeHierarchyFolder = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
		specTypeHierarchyFolder.createAttribute("IDENTIFIER", "OT-" + suffix);
		specTypeHierarchyFolder.createAttribute("LONG-NAME", "Folder Header and Description " + longNameAddition);
		specTypeHierarchyFolder.createAttribute("DESC", "The actual title text is contained in objects of this type, rather than in the hierarchy node.");
		specTypeHierarchyFolder.createAttribute("LAST-CHANGE", templateTime);
		
		var specTypeHierarchyFolderAttrs = specTypeHierarchyFolder.createChild("SPEC-ATTRIBUTES");
		
		var specTypeHierarchyFolderAttr = specTypeHierarchyFolderAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
		specTypeHierarchyFolderAttr.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + suffix);
		specTypeHierarchyFolderAttr.createAttribute("LAST-CHANGE", templateTime);
		specTypeHierarchyFolderAttr.createAttribute("LONG-NAME", LABEL.Heading);
		specTypeHierarchyFolderAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
		
		specTypeHierarchyFolderAttr = specTypeHierarchyFolderAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
		specTypeHierarchyFolderAttr.createAttribute("IDENTIFIER", "AD-ObjectText-" + suffix);
		specTypeHierarchyFolderAttr.createAttribute("LAST-CHANGE", templateTime);
		specTypeHierarchyFolderAttr.createAttribute("LONG-NAME", LABEL.Description);
		specTypeHierarchyFolderAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
		
		writeCustomPropertyTypes(specTypeHierarchyFolderAttrs, IDENTIFIER.object.plan + "Folder", "folder", categoryID);
		
		writeReqIFUserAttributeType(specTypeHierarchyFolderAttrs, suffix);
	});
})();

(function(){
	// IssueSets
	
	var allCategories = cockpit.dataModelProvider.getCategoryIDsForObjectType("gm.issuemodule2.issueSet");
	allCategories.add(0, null); // adding default category
	
	allCategories.toArray().forEach(function(categoryID) {
		var categoryIDRif;
		var longNameAddition;
		if (categoryID == null) {
			categoryIDRif = "";
			longNameAddition = "with default category";
		} else {
			categoryIDRif = "-" + categoryID;
			longNameAddition = "with category '" + cockpit.dataModelProvider.getDisplaynameOfCategory("gm.issuemodule2.issueSet", categoryID) + "'";
		}
		
		var suffix = IDENTIFIER.object.issue + "Folder" + categoryIDRif;
	
		var specTypeHierarchyFolder = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
		specTypeHierarchyFolder.createAttribute("IDENTIFIER", "OT-" + suffix);
		specTypeHierarchyFolder.createAttribute("LONG-NAME", "Folder Header and Description " + longNameAddition);
		specTypeHierarchyFolder.createAttribute("DESC", "The actual title text is contained in objects of this type, rather than in the hierarchy node.");
		specTypeHierarchyFolder.createAttribute("LAST-CHANGE", templateTime);
		
		var specTypeHierarchyFolderAttrs = specTypeHierarchyFolder.createChild("SPEC-ATTRIBUTES");
		
		var specTypeHierarchyFolderAttr = specTypeHierarchyFolderAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
		specTypeHierarchyFolderAttr.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + suffix);
		specTypeHierarchyFolderAttr.createAttribute("LAST-CHANGE", templateTime);
		specTypeHierarchyFolderAttr.createAttribute("LONG-NAME", LABEL.Heading);
		specTypeHierarchyFolderAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
		
		specTypeHierarchyFolderAttr = specTypeHierarchyFolderAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
		specTypeHierarchyFolderAttr.createAttribute("IDENTIFIER", "AD-ObjectText-" + suffix);
		specTypeHierarchyFolderAttr.createAttribute("LAST-CHANGE", templateTime);
		specTypeHierarchyFolderAttr.createAttribute("LONG-NAME", LABEL.Description);
		specTypeHierarchyFolderAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
		
		writeCustomPropertyTypes(specTypeHierarchyFolderAttrs, IDENTIFIER.object.issue + "Folder", "gm.issuemodule2.issueSet", categoryID);
		
		writeReqIFUserAttributeType(specTypeHierarchyFolderAttrs, suffix);
	});
})();

(function(){
	// RequirementSets
	
	var allCategories = cockpit.dataModelProvider.getCategoryIDsForObjectType("gm.requirementsmodule3.requirementSet");
	allCategories.add(0, null); // adding default category
	
	allCategories.toArray().forEach(function(categoryID) {
		var categoryIDRif;
		var longNameAddition;
		if (categoryID == null) {
			categoryIDRif = "";
			longNameAddition = "with default category";
		} else {
			categoryIDRif = "-" + categoryID;
			longNameAddition = "with category '" + cockpit.dataModelProvider.getDisplaynameOfCategory("gm.requirementsmodule3.requirementSet", categoryID) + "'";
		}
		
		var suffix = IDENTIFIER.object.requirement + "Folder" + categoryIDRif;
	
		var specTypeHierarchyFolder = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
		specTypeHierarchyFolder.createAttribute("IDENTIFIER", "OT-" + suffix);
		specTypeHierarchyFolder.createAttribute("LONG-NAME", "Folder Header and Description " + longNameAddition);
		specTypeHierarchyFolder.createAttribute("DESC", "The actual title text is contained in objects of this type, rather than in the hierarchy node.");
		specTypeHierarchyFolder.createAttribute("LAST-CHANGE", templateTime);
		
		var specTypeHierarchyFolderAttrs = specTypeHierarchyFolder.createChild("SPEC-ATTRIBUTES");
		
		var specTypeHierarchyFolderAttr = specTypeHierarchyFolderAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
		specTypeHierarchyFolderAttr.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + suffix);
		specTypeHierarchyFolderAttr.createAttribute("LAST-CHANGE", templateTime);
		specTypeHierarchyFolderAttr.createAttribute("LONG-NAME", LABEL.Heading);
		specTypeHierarchyFolderAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
		
		specTypeHierarchyFolderAttr = specTypeHierarchyFolderAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
		specTypeHierarchyFolderAttr.createAttribute("IDENTIFIER", "AD-ObjectText-" + suffix);
		specTypeHierarchyFolderAttr.createAttribute("LAST-CHANGE", templateTime);
		specTypeHierarchyFolderAttr.createAttribute("LONG-NAME", LABEL.Description);
		specTypeHierarchyFolderAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
		
		writeCustomPropertyTypes(specTypeHierarchyFolderAttrs, IDENTIFIER.object.requirement + "Folder", "gm.requirementsmodule3.requirementSet", categoryID);
		
		writeReqIFUserAttributeType(specTypeHierarchyFolderAttrs, suffix);
	});
})();

//SpecType for the Project and ProjectInfo
(function(){
	var suffix = "Project";
	
	var specType = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
	specType.createAttribute("IDENTIFIER", "OT-" + suffix);
	specType.createAttribute("LONG-NAME", "Project");
	specType.createAttribute("LAST-CHANGE", templateTime);
	
	var specTypeAttributes = specType.createChild("SPEC-ATTRIBUTES");
	
	var specTypeAttribute = specTypeAttributes.createChild("ATTRIBUTE-DEFINITION-STRING");
	specTypeAttribute.createAttribute("IDENTIFIER", "AD-ObjectHeading-" + suffix);
	specTypeAttribute.createAttribute("LAST-CHANGE", templateTime);
	specTypeAttribute.createAttribute("LONG-NAME", LABEL.Heading);
	specTypeAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
	
	specTypeAttribute = specTypeAttributes.createChild("ATTRIBUTE-DEFINITION-XHTML");
	specTypeAttribute.createAttribute("IDENTIFIER", "AD-ObjectText-" + suffix);
	specTypeAttribute.createAttribute("LAST-CHANGE", templateTime);
	specTypeAttribute.createAttribute("LONG-NAME", LABEL.Description);
	specTypeAttribute.createChild("TYPE").createChild("DATATYPE-DEFINITION-XHTML-REF").createText("DT-FormattedText");
})();

(function(){
	// Generic hierarchy folders for non-Cockpit objects
	var specTypeHierarchyFolder = rifSpecTypes.createChild("SPEC-OBJECT-TYPE");
	specTypeHierarchyFolder.createAttribute("IDENTIFIER", "OT-Folder");
	specTypeHierarchyFolder.createAttribute("LONG-NAME", "Folder Header and Description");
	specTypeHierarchyFolder.createAttribute("DESC", "The actual title text is contained in objects of this type, rather than in the hierarchy node.");
	specTypeHierarchyFolder.createAttribute("LAST-CHANGE", templateTime);
	
	var specTypeHierarchyFolderAttrs = specTypeHierarchyFolder.createChild("SPEC-ATTRIBUTES");
	
	var specTypeHierarchyFolderAttr = specTypeHierarchyFolderAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
	specTypeHierarchyFolderAttr.createAttribute("IDENTIFIER", "AD-ObjectHeading-Folder");
	specTypeHierarchyFolderAttr.createAttribute("LAST-CHANGE", templateTime);
	specTypeHierarchyFolderAttr.createAttribute("LONG-NAME", LABEL.Heading);
	specTypeHierarchyFolderAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-String");
	
	specTypeHierarchyFolderAttr = specTypeHierarchyFolderAttrs.createChild("ATTRIBUTE-DEFINITION-STRING");
	specTypeHierarchyFolderAttr.createAttribute("IDENTIFIER", "AD-ObjectText-Folder");
	specTypeHierarchyFolderAttr.createAttribute("LAST-CHANGE", templateTime);
	specTypeHierarchyFolderAttr.createAttribute("LONG-NAME", LABEL.Description);
	specTypeHierarchyFolderAttr.createChild("TYPE").createChild("DATATYPE-DEFINITION-STRING-REF").createText("DT-Text");
	
	writeReqIFUserAttributeType(specTypeHierarchyFolderAttrs, "Folder");
})();

// SpecTypes for Relations
(function(){
	var relationTypesToCreate =	[
		IDENTIFIER.relation.RequirementReferencesRequirement,
		IDENTIFIER.relation.ModelElementLinksRequirement,
		IDENTIFIER.relation.ModelElementContainsModelElement,
		IDENTIFIER.relation.PlanShowsModelElement,
		IDENTIFIER.relation.IssueLinksRequirement,
		IDENTIFIER.relation.IssueLinksModelElement,
		IDENTIFIER.relation.IssueLinksIssue,
		IDENTIFIER.relation.CommentAnnotatesObject,
		IDENTIFIER.relation.NewRequirement,
		IDENTIFIER.relation.NewIssue,
		IDENTIFIER.relation.NonCockpitRelation
	];
	
	relationTypesToCreate.forEach(function(singleRelationType) {
		var specTypeRelation = rifSpecTypes.createChild("SPEC-RELATION-TYPE");
		specTypeRelation.createAttribute("IDENTIFIER", "RT-" + singleRelationType);
		specTypeRelation.createAttribute("LAST-CHANGE", templateTime);	
	});
})();

// SpecTypes for hierarchy roots and folders
(function(){
	var specTypeHierarchyRoot = rifSpecTypes.createChild("SPECIFICATION-TYPE");
	specTypeHierarchyRoot.createAttribute("IDENTIFIER", "HT-Any-hierarchy-root");
	specTypeHierarchyRoot.createAttribute("LONG-NAME", "A Hierarchy Root");
	specTypeHierarchyRoot.createAttribute("DESC", "This is the root node of a hierarchy (tree) of folders and objects.");
	specTypeHierarchyRoot.createAttribute("LAST-CHANGE", templateTime);
})();


/* -------------- Spec Objects --------------- */
LOGGER.debug("Writing REQ-IF SpecObjects.");
var rifSpecObjects = rifContent.createChild("SPEC-OBJECTS");

(function(){
	// Spec Object for the Project
	var specObjectProjectInfo = rifSpecObjects.createChild("SPEC-OBJECT");
	var specObjectProjectInfoIdentifier = composeGUID("ProjectInformation");
	
	specObjectProjectInfo.createAttribute("IDENTIFIER", specObjectProjectInfoIdentifier);
	specObjectProjectInfo.createAttribute("LONG-NAME", global.project.project.getName());
	specObjectProjectInfo.createAttribute("LAST-CHANGE", templateTime);
	specObjectProjectInfo.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-Project");
	
	var specObjectProjectInfoValues = specObjectProjectInfo.createChild("VALUES");
	
	var specObjectProjectInfoAttribute = specObjectProjectInfoValues.createChild("ATTRIBUTE-VALUE-STRING");
	specObjectProjectInfoAttribute.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-Project");
	specObjectProjectInfoAttribute.createAttribute("THE-VALUE", global.project.project.getName());
	
	specObjectProjectInfoAttribute = specObjectProjectInfoValues.createChild("ATTRIBUTE-VALUE-XHTML");
	specObjectProjectInfoAttribute.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-XHTML-REF").createText("AD-ObjectText-Project");
	var specObjectProjectDiv = specObjectProjectInfoAttribute.createChild("THE-VALUE").createChild("xhtml:div");
	
	var file = getPath(global.project.project.getLogo());
	if (file != null) {
		if (includeFileInZip(file.getName(), cockpit.tempImageDirectory, global.zip.folderInZipForFiles)) {
			var logoFileName = file.getName();
			var logoImageType = mimeTypeForExtension(logoFileName);
			if (isImage(logoFileName) && logoImageType != null) {
				var specObjectProjectLogo = specObjectProjectDiv.createChild("xhtml:p").createChild("xhtml:object");
				specObjectProjectLogo.createAttribute("name", logoFileName);
				specObjectProjectLogo.createAttribute("type", logoImageType);
				specObjectProjectLogo.createAttribute("data", file.toString().replace(java.io.File.separator, "/"));
			} else {
				LOGGER.error("Project logo's '" + logoFileName + "' file extension not recognized. You might be using a logo which predates Cockpit 3.1. Please re-set your project logo in Cockpit.");
			}
		}
	}
	
	if (global.project.project.getDescription().length > 0 ) {
		global.project.project.getDescription().forEach(function(line) {
			specObjectProjectDiv.createText(line);
			specObjectProjectDiv.createChild("xhtml:br");
		});
		specObjectProjectDiv.createChild("xhtml:br");
	}
	
	if (global.stakeholders.allStakeholders.length > 0) {
		writeStakeholders(specObjectProjectDiv, global.stakeholders.allStakeholders);
	}
	
	global.allExportedReqIFIdentifier[specObjectProjectInfoIdentifier] = true;
	
	// Folders and Plans
	writeSpecObjectsForFolderContents(rifSpecObjects, null);
	
	(function () {
	
		// Model Elements
		global.project.project.getAllModelElements().toArray().forEach(function(modelElement) {
			if (exportCockpitObject(modelElement)) {
				var reqIfIdentifier = writeSpecObjectModelElement(rifSpecObjects, modelElement);
				
				global.modelElement.exportedModelElementsArray.push(modelElement);
				
				global.allExportedReqIFIdentifier[reqIfIdentifier] = true;
			}
		});
		
		global.modelElement.exportedModelElementSubFoldersArray = [];
		
		// Deviding Model Elements into sub folders, if there are more than a certain threshold
		if (global.modelElement.exportedModelElementsArray.length > OPTION.ModelElementPerSubFolder) {
			
			for (var i = 0; i < global.modelElement.exportedModelElementsArray.length; i = i + OPTION.ModelElementPerSubFolder) {
			
				var firstElemInGroup = global.modelElement.exportedModelElementsArray[i];
				
				var lastElemInGroup;
				if (global.modelElement.exportedModelElementsArray.length - 1 < i + OPTION.ModelElementPerSubFolder) {
					lastElemInGroup = global.modelElement.exportedModelElementsArray[global.modelElement.exportedModelElementsArray.length - 1];
				} else {
					lastElemInGroup = global.modelElement.exportedModelElementsArray[i + OPTION.ModelElementPerSubFolder - 1];
				}
				
				var firstElemInGroupName = firstElemInGroup.getName();
				var firstElemInGroupShortName;
				if (firstElemInGroupName.length() > 4) {
					firstElemInGroupShortName = firstElemInGroupName.substring(0, 3) + "...";
				} else {
					firstElemInGroupShortName = firstElemInGroupName;
				}
				
				var lastElemInGroupName = lastElemInGroup.getName();
				var lastElemInGroupShortName;
				if (lastElemInGroupName.length() > 4) {
					lastElemInGroupShortName = lastElemInGroupName.substring(0, 3) + "...";
				} else {
					lastElemInGroupShortName = lastElemInGroupName;
				}
				
				var longName = firstElemInGroupShortName + " - " + lastElemInGroupShortName;
				var headingName = longName;
				var objectText = "Modelelemente von '" + firstElemInGroupName + "' bis '" + lastElemInGroupName + "'";
				
				var specObjectElemFolder = rifSpecObjects.createChild("SPEC-OBJECT");
				
				var identifier = composeGUID(IDENTIFIER.object.modelElement + "Folder-" + Math.floor(i / OPTION.ModelElementPerSubFolder));
				
				specObjectElemFolder.createAttribute("IDENTIFIER", identifier);
				specObjectElemFolder.createAttribute("LONG-NAME", longName);
				specObjectElemFolder.createAttribute("LAST-CHANGE", templateTime);
				specObjectElemFolder.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-Folder");
		
				var specObjectElemFolderValues = specObjectElemFolder.createChild("VALUES");
				
				var specObjectElemFolderAttribute = specObjectElemFolderValues.createChild("ATTRIBUTE-VALUE-STRING");
				specObjectElemFolderAttribute.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-Folder");
				specObjectElemFolderAttribute.createAttribute("THE-VALUE", headingName);
				
				var specObjectElemFolderAttributeText = specObjectElemFolderValues.createChild("ATTRIBUTE-VALUE-STRING");
				specObjectElemFolderAttributeText.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectText-Folder");
				specObjectElemFolderAttributeText.createAttribute("THE-VALUE", objectText);
				
				writeReqIFUserAttributeValue(specObjectElemFolderValues, "Folder", {"username" : "Co-Pilot"});
				
				global.allExportedReqIFIdentifier[identifier] = true;
			}
		}
		
	}) ();
	
	// Requirement Sets and Requirements
	writeSpecObjectsForRequirementSetContents(rifSpecObjects, null);
	
	// IssueSets and Issues including Comments
	writeSpecObjectsForIssueSetContents(rifSpecObjects, null);
	
	// Spec Object for the project itself
	var specObjectProject = rifSpecObjects.createChild("SPEC-OBJECT");
	var specObjectProjectIdentifier = composeGUID("Project");
	
	specObjectProject.createAttribute("IDENTIFIER", specObjectProjectIdentifier);
	specObjectProject.createAttribute("LONG-NAME", "Project");
	specObjectProject.createAttribute("LAST-CHANGE", actualDateTimeISO);
	specObjectProject.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-Project");
	
	global.allExportedReqIFIdentifier[specObjectProjectIdentifier] = true;
	
	// Spec Object for the model elements chapter title (glossary)
	var specObjectModelElementChapter = rifSpecObjects.createChild("SPEC-OBJECT");
	var specObjectModelElementChapterIdentifier = composeGUID(IDENTIFIER.object.modelElement + "Chapter");
	
	specObjectModelElementChapter.createAttribute("IDENTIFIER", specObjectModelElementChapterIdentifier);
	specObjectModelElementChapter.createAttribute("LONG-NAME", LABEL.GlossaryChapterTitle);
	specObjectModelElementChapter.createAttribute("LAST-CHANGE", templateTime);
	specObjectModelElementChapter.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-Folder");
	
	var specObjectModelElementChapterValues = specObjectModelElementChapter.createChild("VALUES");
	
	var specObjectModelElementChapterAttribute = specObjectModelElementChapterValues.createChild("ATTRIBUTE-VALUE-STRING");
	specObjectModelElementChapterAttribute.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-Folder");
	specObjectModelElementChapterAttribute.createAttribute("THE-VALUE", LABEL.GlossaryChapterTitle);
	
	writeReqIFUserAttributeValue(specObjectModelElementChapterValues, "Folder", {"username" : "Co-Pilot"});
	
	global.allExportedReqIFIdentifier[specObjectModelElementChapterIdentifier] = true;
	
	// Spec Object for the requirements chapter title
	var specObjectRequirementChapter = rifSpecObjects.createChild("SPEC-OBJECT");
	var specObjectRequirementChapterIdentifier = composeGUID(IDENTIFIER.object.requirement + "Chapter");
	
	specObjectRequirementChapter.createAttribute("IDENTIFIER", specObjectRequirementChapterIdentifier);
	specObjectRequirementChapter.createAttribute("LONG-NAME", LABEL.RequirementsChapterTitle);
	specObjectRequirementChapter.createAttribute("LAST-CHANGE", templateTime);
	specObjectRequirementChapter.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-Folder");

	var specObjectRequirementChapterValues = specObjectRequirementChapter.createChild("VALUES");
	
	var specObjectRequirementChapterAttribute = specObjectRequirementChapterValues.createChild("ATTRIBUTE-VALUE-STRING");
	specObjectRequirementChapterAttribute.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-Folder");
	specObjectRequirementChapterAttribute.createAttribute("THE-VALUE", LABEL.RequirementsChapterTitle);
	
	writeReqIFUserAttributeValue(specObjectRequirementChapterValues, "Folder", {"username" : "Co-Pilot"});
	
	global.allExportedReqIFIdentifier[specObjectRequirementChapterIdentifier] = true;
	
	// Spec Object for the issues chapter title
	var specObjectIssueChapter = rifSpecObjects.createChild("SPEC-OBJECT");
	var specObjectIssueChapterIdentifier = composeGUID(IDENTIFIER.object.issue + "Chapter");
	
	specObjectIssueChapter.createAttribute("IDENTIFIER", specObjectIssueChapterIdentifier);
	specObjectIssueChapter.createAttribute("LONG-NAME", LABEL.IssueChapterTitle);
	specObjectIssueChapter.createAttribute("LAST-CHANGE", templateTime);
	specObjectIssueChapter.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-Folder");

	var specObjectIssueChapterValues = specObjectIssueChapter.createChild("VALUES");
	
	var specObjectIssueChapterAttribute = specObjectIssueChapterValues.createChild("ATTRIBUTE-VALUE-STRING");
	specObjectIssueChapterAttribute.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-ObjectHeading-Folder");
	specObjectIssueChapterAttribute.createAttribute("THE-VALUE", LABEL.IssueChapterTitle);
	
	writeReqIFUserAttributeValue(specObjectIssueChapterValues, "Folder", {"username" : "Co-Pilot"});
	
	global.allExportedReqIFIdentifier[specObjectIssueChapterIdentifier] = true;
	
	// Spec Object for exported project revision
	var specObjectRevision = rifSpecObjects.createChild("SPEC-OBJECT");
	var specObjectRevisionIdentifier = composeGUID("Setting-ProjectRevision");
	specObjectRevision.createAttribute("IDENTIFIER", specObjectRevisionIdentifier);
	specObjectRevision.createAttribute("LAST-CHANGE", templateTime);
	specObjectRevision.createChild("TYPE").createChild("SPEC-OBJECT-TYPE-REF").createText("OT-Setting");
	
	var specObjectValues = specObjectRevision.createChild("VALUES");
	
	var specObjectAttribute = specObjectValues.createChild("ATTRIBUTE-VALUE-STRING");
	specObjectAttribute.createChild("DEFINITION").createChild("ATTRIBUTE-DEFINITION-STRING-REF").createText("AD-Value-Setting");
	// The currently commited version in cockpit.
	// Potentially increasing the version by 1 on the ReqIF server, if there was a successfull import into Cockpit in this sync run.
	// This is needed, as the commit happens after the exection of the JavaScript, but the export already exports the new data like comments etc ...
	specObjectAttribute.createAttribute("THE-VALUE", global.project.maxVersion + ((increaseVersionCountByOne) ? 1 : 0));
})();

/* ----------------- Spec Relations -------------------- */
// Relations are created for each object that was exported before
LOGGER.debug("Writing REQ-IF SpecRelations.");

var rifSpecRelations = rifContent.createChild("SPEC-RELATIONS");

(function(){
	var requirement;
	for (var key in global.requirement.exportedRequirements) {
		if (global.requirement.exportedRequirements.hasOwnProperty(key)) {
			requirement = global.requirement.exportedRequirements[key];
			writeRelationRequirementToRequirement(rifSpecRelations, requirement);
			writeRelationRequirementToModelElement(rifSpecRelations, requirement);
			writeRelationNonCockpitLink(rifSpecRelations, requirement);
		}
	}
})();

(function(){
	global.modelElement.exportedModelElementsArray.forEach(function(modelElement) {
		writeRelationModelElementContainsModelElement(rifSpecRelations, modelElement);
	});
})();

(function(){
	var plan;
	for (var key in global.plan.exportedPlans) {
		if (global.plan.exportedPlans.hasOwnProperty(key)) {
			plan = global.plan.exportedPlans[key];
			writeRelationPlanShowsModelElement(rifSpecRelations, plan);
		}
	}
})();

(function(){
	var issue;
	for (var key in global.issue.exportedIssues) {
		if (global.issue.exportedIssues.hasOwnProperty(key)) {
			issue = global.issue.exportedIssues[key];
			writeRelationIssueConcernsModelElement(rifSpecRelations, issue);
			writeRelationIssueConcernsRequirement(rifSpecRelations, issue);
			writeRelationIssueConcernsIssue(rifSpecRelations, issue);
			writeRelationNonCockpitLink(rifSpecRelations, issue);
		}
	}
})();

/* ----------------- Spec Hierarchies -------------------- */
LOGGER.debug("Writing REQ-IF SpecHierarchy.");

var rifHierarchyRoots = rifContent.createChild("SPECIFICATIONS");

(function(){
	var hierarchyRoot;
	var hierarchyRootChilden;
	var hierarchyProject;
	var hierarchyProjectChildren;
	var hierarchyChild;
	
	// Hierarchy root for all objects
	hierarchyRoot = rifHierarchyRoots.createChild("SPECIFICATION");
	hierarchyRoot.createAttribute("IDENTIFIER", global.project.reqIfIdentifier + "-Specification");
	hierarchyRoot.createAttribute("LAST-CHANGE", templateTime);
	hierarchyRoot.createChild("TYPE").createChild("SPECIFICATION-TYPE-REF").createText("HT-Any-hierarchy-root");
	hierarchyRootChilden = hierarchyRoot.createChild("CHILDREN");
	
	hierarchyProject = hierarchyRootChilden.createChild("SPEC-HIERARCHY");
	hierarchyProject.createAttribute("IDENTIFIER", "ProjInfo-h1-1");
	hierarchyProject.createAttribute("LAST-CHANGE", templateTime);
	hierarchyProject.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(composeGUID("ProjectInformation"));
	
	hierarchyProjectChildren = hierarchyProject.createChild("CHILDREN");
	
	// Plans and folders
	var counter = new Object();
	counter.folder = 0;
	counter.element = 0;
	
	writeSpecHierarchyFolderContents(hierarchyProjectChildren, null, 0, counter);
	
	// Root for Issues
	hierarchyChild = hierarchyProjectChildren.createChild("SPEC-HIERARCHY");
	hierarchyChild.createAttribute("IDENTIFIER", IDENTIFIER.object.issue + "RootFolder");
	hierarchyChild.createAttribute("LAST-CHANGE", templateTime);
	hierarchyChild.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(composeGUID(IDENTIFIER.object.issue + "Chapter"));
	
	var hierarchyChildChildren = hierarchyChild.createChild("CHILDREN");
	
	writeSpecHierarchyIssueSetContents(hierarchyChildChildren, null, 0, counter);
	
	// Root for Requirements
	hierarchyChild = hierarchyProjectChildren.createChild("SPEC-HIERARCHY");
	hierarchyChild.createAttribute("IDENTIFIER", IDENTIFIER.object.requirement + "RootFolder");
	hierarchyChild.createAttribute("LAST-CHANGE", templateTime);
	hierarchyChild.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(composeGUID(IDENTIFIER.object.requirement + "Chapter"));
	
	hierarchyChildChildren = hierarchyChild.createChild("CHILDREN");
	
	counter = new Object();
	counter.folder = 0;
	counter.element = 0;
	
	writeSpecHierarchyRequirementSetContents(hierarchyChildChildren, null, 0, counter);
	
	// Root for Model Element (Glossary) including all Model Elements
	hierarchyChild = hierarchyProjectChildren.createChild("SPEC-HIERARCHY");
	hierarchyChild.createAttribute("IDENTIFIER", IDENTIFIER.object.modelElement + "Folder-h1-1");
	hierarchyChild.createAttribute("LAST-CHANGE", templateTime);
	hierarchyChild.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(composeGUID(IDENTIFIER.object.modelElement + "Chapter"));
	hierarchyChildChildren = hierarchyChild.createChild("CHILDREN");
	
	(function () {
		var elemCounter = 0;
		var currentParent = hierarchyChildChildren;
		global.modelElement.exportedModelElementsArray.forEach(function (modelElement) {
						
			if (global.modelElement.exportedModelElementsArray.length > OPTION.ModelElementPerSubFolder && elemCounter % OPTION.ModelElementPerSubFolder === 0) {
				
				var subFolderCounter = Math.floor(elemCounter / OPTION.ModelElementPerSubFolder);
				
				var subFolderRef = hierarchyChildChildren.createChild("SPEC-HIERARCHY");
				subFolderRef.createAttribute("IDENTIFIER", IDENTIFIER.object.modelElement + "Folder-h2-" + subFolderCounter);
				subFolderRef.createAttribute("LAST-CHANGE", templateTime);
				subFolderRef.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(composeGUID(IDENTIFIER.object.modelElement + "Folder-" + subFolderCounter));
				currentParent = subFolderRef.createChild("CHILDREN");
			}
			
			var hierarchyChildChildrenHierachy = currentParent.createChild("SPEC-HIERARCHY");
			var modelElementGUID = getRifIdentifierForModelElement(modelElement);
			hierarchyChildChildrenHierachy.createAttribute("IDENTIFIER", "Hierarchy-" + modelElementGUID);
			hierarchyChildChildrenHierachy.createAttribute("LAST-CHANGE", templateTime);
			hierarchyChildChildrenHierachy.createChild("OBJECT").createChild("SPEC-OBJECT-REF").createText(modelElementGUID);
			
			elemCounter++;
		});
	})();
})();

/* FOR DEBUGGING ONLY! */
// var zipPath = cockpit.logDir + java.io.File.separator;	// Variant 1 with the path being the for each export uniquely created log directory 
// var zipPath = "D:\\Debugging\\CoPilot";					// Variant 2 with fixed path
// var fileout = new java.io.FileOutputStream(new java.io.File(zipPath + "CoPilotExport-" + new Date().getTime() + ".zip"));
// zipContents(fileout, doc.doc, global.zip.filesToZip);
// fileout.close();
/* FOR DEBUGGING ONLY! */


/* ======== Post the Data to the Server ========= */
LOGGER.info("Starting export to ReqIF Server");
(function(){
	// Setup connection to server for import
	LOGGER.debug("Sending contents to the server.");
	var importID = null;
	
	var importResponse = sendRequestToServer(
				global.urlbuilder.getReqIFURL("import"),
				OPTION.ReqIFUsername,
				OPTION.ReqIFPassword,
				{
					"requestMethod" : "POST",
					"runBeforeRequest" : function(httpurlconnection) {
						httpurlconnection.setRequestProperty("Content-Type", "application/octet-stream");
						httpurlconnection.setRequestProperty("Accept", "application/xml");
						httpurlconnection.setDoOutput(true);
						httpurlconnection.setChunkedStreamingMode(1024*1024); // 1MB chunks
					},
					"dataFunction" : function(outputstream) {
						// Zip the contents and send it via POST request
						zipContents(outputstream, doc.doc, global.zip.filesToZip);
						outputstream.flush();
						outputstream.close();
					},
					// get import job ID and start import
					"runAfterRequest" : function(httpurlconnection) {
						importID = httpurlconnection.getHeaderField("location");
					}
				}
	);
	
	if (importResponse.code !== 201) {
		throw new Error("ReqIF Server failed to import. Code: '" + importResponse.code + "'.");
	}
	
	assert(importID, "Location header not found in import response");

	LOGGER.debug("Triggering import on server.");
	var startTime = new Date();
	var triggerResponse = sendRequestToServer(global.urlbuilder.getReqIFURL(importID), OPTION.ReqIFUsername, OPTION.ReqIFPassword, {"requestMethod" : "PUT"});
	if (triggerResponse.code !== 202) {
		throw new Error("Failed to trigger import. Code: '" + importResponse.code + "'.");
	}

	LOGGER.debug("Waiting for import to finish.");
	var importStatus;
	do {
		java.lang.Thread.sleep(10000);
		importStatus = sendRequestToServer(global.urlbuilder.getReqIFURL(importID), OPTION.ReqIFUsername, OPTION.ReqIFPassword, null);
	} while (importStatus.code === 200 && importStatus.content === "IMPORTING");
	
	var endTime = new Date();
	
	if (importStatus.code === 200 && importStatus.content === "SUCCESSFUL") {
		LOGGER.info("Export successfully done with status '" + importStatus.content + "' in " + ((endTime - startTime) / 1000) + " seconds.");
	} else {
		throw Error("Export not done. Ended with code '" + importStatus.code + "' and status '" + importStatus.content + "'");
	}
})();
}